/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.runtime.logger;

import java.text.MessageFormat;

import org.apache.commons.logging.Log;

import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.runtime.AbstractEventProcessorListener;
import com.evelopers.unimod.runtime.StateMachineConfig;
import com.evelopers.unimod.runtime.StateMachinePath;
import com.evelopers.unimod.runtime.context.StateMachineContext;

/**
 * Simple pluggable logger unit.
 *
 * @author Vadim Gurov
 * @version $Revision: 1$
 */
public class SimpleLogger extends AbstractEventProcessorListener {
	private Log log;

    private static final MessageFormat EVENT_PROCESSING_STARTED_MSG = new MessageFormat("Start event [{0}] processing. In state [{1}:{2}]");
    private static final MessageFormat EVENT_PROCESSING_FINISHED_MSG = new MessageFormat("Finish event [{0}] processing. In state [{1}:{2}]");
    private static final MessageFormat STATE_MACHINE_CAME_TO_FINAL_STATE_MSG = new MessageFormat("State machine came to final state [{0}:{1}]");
    private static final MessageFormat BEFORE_SUBMACHINES_EXECUTION_MSG = new MessageFormat("Start state machines included to state [{0}:{1}] execution");
    private static final MessageFormat AFTER_SUBMACHINES_EXECUTION_MSG = new MessageFormat("Finish state machines included to state [{0}:{1}] execution");
    private static final MessageFormat BEFORE_ON_ENTER_ACTION_EXECUTION_MSG = new MessageFormat("Start on-enter action [{0}] execution");
    private static final MessageFormat AFTER_ON_ENTER_ACTION_EXECUTION_MSG = new MessageFormat("Finish on-enter action [{0}] execution");
    private static final MessageFormat BEFORE_OUTPUT_ACTION_EXECUTION_MSG = new MessageFormat("Start output action [{0}] execution");
    private static final MessageFormat AFTER_OUTPUT_ACTION_EXECUTION_MSG = new MessageFormat("Finish output action [{0}] execution");
    private static final MessageFormat BEFORE_INPUT_ACTION_EXECUTION_MSG = new MessageFormat("Start input action [{0}] calculation");
    private static final MessageFormat AFTER_INPUT_ACTION_EXECUTION_MSG = new MessageFormat("Finish input action [{0}] calculation. Its value is [{1}]");
    private static final MessageFormat TRANSITION_CANDIDATE_MSG = new MessageFormat("Try transition [{0}]");
    private static final MessageFormat TRANSITION_FOUND_MSG = new MessageFormat("Transition to go found [{0}]");
    private static final MessageFormat TRANSITION_NOT_FOUND_MSG = new MessageFormat("No transition defined for event [{0}] from state");

    public SimpleLogger(Log log) {
		this.log = log;
	}

    public void eventProcessingStarted(StateMachineContext context, Event e, StateMachinePath path, StateMachineConfig c) {
        log.info(EVENT_PROCESSING_STARTED_MSG.format(new Object[]{e, path, c}));
    }

    public void eventProcessingFinished(StateMachineContext context, Event e, StateMachinePath path, StateMachineConfig c) {
        log.info(EVENT_PROCESSING_FINISHED_MSG.format(new Object[]{e, path, c}));
    }

    public void stateMachineCameToFinalState(StateMachineContext context, StateMachinePath path, StateMachineConfig config) {
        log.info(STATE_MACHINE_CAME_TO_FINAL_STATE_MSG.format(new Object[]{path, config}));
    }

    public void beforeSubmachinesExecution(StateMachineContext context, Event event, StateMachinePath path, String config) {
        log.info(BEFORE_SUBMACHINES_EXECUTION_MSG.format(new Object[]{path, config}));
    }

    public void afterSubmachinesExecution(StateMachineContext context, Event event, StateMachinePath path, String state) {
        log.info(AFTER_SUBMACHINES_EXECUTION_MSG.format(new Object[]{path, state}));
    }

    public void beforeOnEnterActionExecution(StateMachineContext context, 
            StateMachinePath path, String stateName, String outputAction) {
        log.info(BEFORE_ON_ENTER_ACTION_EXECUTION_MSG.format(new Object[]{outputAction}));
    }

    public void afterOnEnterActionExecution(StateMachineContext context, 
            StateMachinePath path, String stateName, String outputAction) {
        log.info(AFTER_ON_ENTER_ACTION_EXECUTION_MSG.format(new Object[]{outputAction}));
    }

    public void beforeOutputActionExecution(StateMachineContext context, 
            StateMachinePath path, String transition, String outputAction) {
        log.info(BEFORE_OUTPUT_ACTION_EXECUTION_MSG.format(new Object[]{outputAction}));
    }

    public void afterOutputActionExecution(StateMachineContext context, 
            StateMachinePath path, String transition, String outputAction) {
        log.info(AFTER_OUTPUT_ACTION_EXECUTION_MSG.format(new Object[]{outputAction}));
    }

    public void beforeInputActionExecution(StateMachineContext context, 
            StateMachinePath path, String transition, String inputAction) {
        log.info(BEFORE_INPUT_ACTION_EXECUTION_MSG.format(new Object[]{inputAction}));
    }

    public void afterInputActionExecution(StateMachineContext context, 
            StateMachinePath path, String transition, String inputAction, String value) {
        log.info(AFTER_INPUT_ACTION_EXECUTION_MSG.format(new Object[]{inputAction, value}));
    }

    public void transitionCandidate(StateMachineContext context, StateMachinePath path, String state, Event event, String transition) {
        log.debug(TRANSITION_CANDIDATE_MSG.format(new Object[]{transition}));
    }

    public void transitionFound(StateMachineContext context, StateMachinePath path, String state, Event event, String transition) {
        log.info(TRANSITION_FOUND_MSG.format(new Object[]{transition}));
    }

    public void transitionNotFound(StateMachineContext context, StateMachinePath path, String state, Event event) {
        log.error(TRANSITION_NOT_FOUND_MSG.format(new Object[]{event, state}));
    }
}