/*
 * Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 * This is open source software; you can use, redistribute and/or modify 
 * it under the terms of the Open Software Licence v 2.1 as published by the Open 
 * Source Initiative.
 *
 * You should have received a copy of the Open Software Licence along with this
 * application; if not, contact the Open Source Initiative (http://opensource.org).
 */
package com.evelopers.unimod.transform.source;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.Writer;
import java.util.Calendar;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.collections.ExtendedProperties;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.FieldMethodizer;
import org.apache.velocity.app.Velocity;
import org.apache.velocity.app.VelocityEngine;
import org.apache.velocity.runtime.RuntimeServices;
import org.apache.velocity.runtime.log.LogSystem;

import com.evelopers.common.util.helper.StringHelper;
import com.evelopers.unimod.contract.CoreContract;
import com.evelopers.unimod.core.stateworks.ControlledObjectHandler;
import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.core.stateworks.Model;
import com.evelopers.unimod.core.stateworks.StateType;
import com.evelopers.unimod.parser.ActionCollector;
import com.evelopers.unimod.parser.Infix;
import com.evelopers.unimod.transform.TransformException;
import com.evelopers.unimod.transform.xml.XMLToModel;

/**
 * Generate source file from given templates.
 * Requires velocity templates. 
 * 
 * @author vgurov
 */
public class ModelToSource {
    
    private VelocityEngine ve;
    private LogSystem veloLog; 
    private final Log log;
    
    private ModelToSource(Log logger) {
        this.log = logger;
        if (log != null) {
            this.veloLog = new LogSystem() {
                public void init(RuntimeServices rs) throws Exception {}

                public void logVelocityMessage(int level, String message) {
                    switch (level) {
                	case LogSystem.DEBUG_ID:
                	    log.debug(message);
                	    break;
                	case LogSystem.ERROR_ID:
                	    log.error(message);
                	    break;
                	case LogSystem.INFO_ID:
                	    log.info(message);
                	    break;
                	case LogSystem.WARN_ID:
                	    log.warn(message);
                	    break;
                    }
                }
            };
        }
    }
    
    public static synchronized ModelToSource create(Log log) {
        return new ModelToSource(log);
    }

    private void generate(Template template, Model model, Writer w, Map additionalParameters) throws Exception {

        VelocityContext vc = new VelocityContext();

        for (Iterator keys = additionalParameters.keySet().iterator(); keys.hasNext();) {
            Object key = keys.next();
            vc.put(key.toString(), additionalParameters.get(key));
        }
        
        vc.put("model", model);
        vc.put("coreContract", CoreContract.create());
        vc.put("tool", Tool.create());
        vc.put("infix", new Infix());
        vc.put("stateType", new FieldMethodizer(StateType.NORMAL));
        vc.put("eventAny", Event.ANY);
        vc.put("actionCollector", new ActionCollector());
        vc.put("boolean", new Boolean(true));
        vc.put("integer", new Integer(0));
        vc.put("controlledObjectHandlerClass", ControlledObjectHandler.class);
        vc.put("generationTime", Calendar.getInstance().getTime());
        
        template.merge(vc, w);
        
        w.flush();
        w.close();
    }

    private void generate(Template template, Model model, File f, Map additionalParameters) throws Exception {
        if (f.getAbsoluteFile().getParentFile() != null) {
            f.getAbsoluteFile().getParentFile().mkdirs();
        }
        
        generate(template, model, new FileWriter(f), additionalParameters);
    }

    private void initVelocity(String key, String value, ClassLoader classLoader) throws Exception {
        ve = new VelocityEngine();
        
        Properties p = new Properties();
        p.load(ModelToSource.class.getResourceAsStream("velocity.properties"));
        
        if (key != null) {
            p.setProperty(key, value);
        }
        
        if (veloLog != null) {
            ve.setProperty(Velocity.RUNTIME_LOG_LOGSYSTEM, veloLog);
        }
        
        ExtendedProperties extendedProperties = ExtendedProperties.convertProperties(p);
        extendedProperties.addProperty("explicitclass.resource.loader." +
        		ExplicitClasspathResourceLoader.CLASSLOADER, classLoader);
		ve.setExtendedProperties(extendedProperties);
        ve.init();
    }
    
    /**
     * Generates source file based on given template from given jar file
     * 
     * @param jarFile jar fle with templates
     * @param templateName templte name inside jar file 
     * @param model model to generate source from 
     * @param outputFile output file
     * @param additionalParameters additional templates parameters
     * @throws TransformException
     */
    public synchronized void generate(File jarFile, String templateName, Model model, File outputFile, Map additionalParameters) throws TransformException {
        try {
            initVelocity("jar.resource.loader.path", "jar:file:" + jarFile.getAbsolutePath(), null);
            
            Template template = ve.getTemplate(templateName);
            generate(template, model, outputFile, additionalParameters);
        } catch (Exception e) {
            throw new TransformException(e);
        }
    }

    /**
     * Generates source file based on given template from class path
     * 
     * @param templateResourceName template file in classpath
     * @param model
     * @param outputFile
     * @param additionalParameters
     * @throws TransformException
     */
    public synchronized void generate(String templateResourceName, Model model, File outputFile, Map additionalParameters, ClassLoader classLoader) throws TransformException {
        try {
            initVelocity(null, null, classLoader);
            
            Template template = ve.getTemplate(templateResourceName);
            generate(template, model, outputFile, additionalParameters);
        } catch (Exception e) {
            throw new TransformException(e);
        }
    }

    public synchronized void generate(String templateResourceName, Model model, Writer w, Map additionalParameters, ClassLoader classLoader) throws TransformException {
        try {
            initVelocity(null, null, classLoader);
            
            Template template = ve.getTemplate(templateResourceName);
            generate(template, model, w, additionalParameters);
        } catch (Exception e) {
            throw new TransformException(e);
        }
    }
    
    /**
     * Generates source file based on given template file 
     * 
     * @param templateFile template file
     * @param model
     * @param outputFile
     * @param additionalParameters
     * @throws TransformException
     */
    public synchronized void generate(File templateFile, Model model, File outputFile, Map additionalParameters) throws TransformException {
        try {
            initVelocity("file.resource.loader.path", templateFile.getParentFile().getAbsolutePath(), null);
            
            Template template = ve.getTemplate(templateFile.getName());
            generate(template, model, outputFile, additionalParameters);
        } catch (Exception e) {
            throw new TransformException(e);
        }
    }

    public static void main(String[] args) throws Exception {
        
        if (args.length < 3) {
            System.out.println("Usage: ModelToSource <model xml> <template file> <output file> [key=value]*");
            
            return;
        }
        
        File model = new File(args[0]); 
        if (!model.exists()) {
            System.out.println("Can't find model: " + model);
        }
        
        File template = new File(args[1]);
        if (!template.exists()) {
            System.out.println("Can't find template: " + template);
        }
        
        File out = new File(args[2]);
        
        ModelToSource ms = ModelToSource.create(LogFactory.getLog(ModelToSource.class));
        Map p = new HashMap();
        
        if (args.length > 3) {
            for (int i = 3; i < args.length; i++) {
                String[] k = StringHelper.splitString(args[i], "=");
                
                if (k.length < 2) {
                    System.out.println("Can't parse key=value pair: " + args[i]);
                    return;
                }
                
                System.out.println("add parameter [" + k[0] + "=" + k[1] + "]");
                p.put(k[0], k[1]);
            }
        }

        try {
            ms.generate(
                    template, 
                    XMLToModel.loadAndCompile(new FileInputStream(model)), 
                    out, p);
        } catch (TransformException e) {
            e.getParentException().printStackTrace();
        }
    }
    
}
