/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.transform.xml;

import com.evelopers.unimod.transform.TransformException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import java.io.OutputStream;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

/**
 * XML Helper. Has method to save XML-DOM to stream.
 * 
 * @author vgurov
 * @version $Revision: 1$
 */
public class XMLHelper {

    public static void save(Document d, OutputStream out, String publicId, String systemId) throws TransformException {
        try {
            Transformer t = TransformerFactory.newInstance().newTransformer();

            t.setOutputProperty(OutputKeys.DOCTYPE_PUBLIC, publicId);
            t.setOutputProperty(OutputKeys.DOCTYPE_SYSTEM, systemId);
            t.setOutputProperty(OutputKeys.INDENT, "no");
            t.setOutputProperty(OutputKeys.METHOD, "xml");
            t.setOutputProperty(OutputKeys.ENCODING, "UTF-8");
            t.setOutputProperty(OutputKeys.VERSION, "1.0");

            addIndents(d);

            t.transform(new DOMSource(d), new StreamResult(out));
        } catch (Exception e) {
            throw new TransformException(e);
        }
    }

    public static void addIndents(Document document) {
        formatTextNode(document.getLastChild());
        indentNode(document.getLastChild(), 1);
    }

    public static void removeNullAttributes(Node node) {
    	switch (node.getNodeType()) {
			case Node.DOCUMENT_NODE:
				removeNullAttributes(node.getLastChild());
				break;
			case Node.ELEMENT_NODE:
				Element element = ((Element)node);
				NamedNodeMap attributes = element.getAttributes();
				if (attributes != null) {
					for (int i = 0; i < attributes.getLength(); i++) {
						Node attribute = attributes.item(i);
						if (attribute.getNodeValue() == null) {
							element.removeAttribute(attribute.getNodeName());
							i --;
						}
					}
				}
				NodeList children = element.getChildNodes();
				for (int i = 0; i < children.getLength(); i++) {
					removeNullAttributes(children.item(i));
				}
				break;
		}
    }
    
    private static void indentNode(Node node, int level) {
        if (node.getNodeType() == Node.ELEMENT_NODE && node.getChildNodes().getLength() > 0) {
            StringBuffer buffer = new StringBuffer("\n");
            for (int i = 0; i < level - 1; i++) {
                buffer.append("  ");
            }
            String endIndent = buffer.toString();
            buffer.append("  ");
            String childIndent = buffer.toString();

            NodeList children = node.getChildNodes();
            List l = new LinkedList();
            for (int i = 0; i < children.getLength(); i++) {
                l.add(children.item(i));
            }
            for (Iterator i = l.iterator(); i.hasNext();) {
                Node child = (Node) i.next();
                Node text = node.getOwnerDocument().createTextNode(childIndent);
                node.insertBefore(text, child);
            }
            Node text = node.getOwnerDocument().createTextNode(endIndent);
            node.appendChild(text);

            for (Iterator i = l.iterator(); i.hasNext();) {
                Node child = (Node) i.next();
                indentNode(child, level + 1);
            }
        }
    }

    private static void formatTextNode(Node node) {
        if (node.getNodeType() == Node.TEXT_NODE) {
            String value = node.getNodeValue().trim();
            node.setNodeValue(value);
            if (value.length() <= 0) {
                node.getParentNode().removeChild(node);
                return;
            }
        }
        List list = new LinkedList();
        for (int i = 0; i < node.getChildNodes().getLength(); i++) {
            list.add(node.getChildNodes().item(i));
        }
        for (Iterator i = list.iterator(); i.hasNext();) {
            Node node1 = (Node) i.next();
            formatTextNode(node1);
        }
    }
}
