/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.transform.xml;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.apache.commons.digester.AbstractObjectCreationFactory;
import org.apache.commons.digester.Digester;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.xml.sax.Attributes;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.XMLReader;

import com.evelopers.unimod.compilation.DefaultCompilationListener;
import com.evelopers.unimod.compilation.StateMachineCompiler;
import com.evelopers.unimod.core.stateworks.ClassElement;
import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.core.stateworks.EventProviderHandler;
import com.evelopers.unimod.core.stateworks.Guard;
import com.evelopers.unimod.core.stateworks.Model;
import com.evelopers.unimod.core.stateworks.State;
import com.evelopers.unimod.core.stateworks.StateMachine;
import com.evelopers.unimod.core.stateworks.StateType;
import com.evelopers.unimod.core.stateworks.Transition;
import com.evelopers.unimod.transform.TransformException;

/**
 * @author Maxim Mazin
 * @version Revision: 1
 */
public class XMLToModel implements XML, ErrorHandler {
	
	protected static final Log l = LogFactory.getLog(XMLToModel.class);
	
    protected Map unresolvedMachines = new HashMap();
    protected Model model;
    protected StateMachine currentMachine = null;

    public static XMLToModel create() {
        return new XMLToModel();
    }

    protected XMLToModel() {
    }

    public Model transform(InputStream source) throws TransformException {
        return transform(source, null);
    }

    public Model transform(InputStream source, XMLReader xmlReader) throws TransformException {
        return _transform(source, xmlReader);
    }

    protected Model _transform(InputStream xml) throws TransformException {
        return _transform(xml, null);
    }

    protected Model _transform(InputStream xml, XMLReader xmlReader) throws TransformException {
        try {
            Digester digester = xmlReader != null ? new Digester(xmlReader) : new Digester();
            URL url = getClass().getClassLoader().getResource(DTD_RESOURCE);
            digester.register(PUBLIC_ID, url.toString());
            digester.setValidating(true);
            digester.setErrorHandler(this);
            
            // model
            digester.addFactoryCreate(
                    Path.MODEL, new AbstractObjectCreationFactory() {
                        public Object createObject(Attributes attributes) throws Exception {
                            String name = attributes.getValue(Attribute.NAME);
                            model = createModel(name);
                            
                            return model;
                        }
                    });

            // controlledObject*
            digester.addFactoryCreate(
                    Path.CONTROLLED_OBJECT, new AbstractObjectCreationFactory() {
                        public Object createObject(Attributes attributes) throws Exception {
                            String name = attributes.getValue(Attribute.NAME);
                            String implClassName = attributes.getValue(Attribute.CLASS);

                            return model.createControlledObjectHandler(name, implClassName);
                        }
                    });

            // eventProvider*
            digester.addFactoryCreate(
                    Path.EVENT_PROVIDER, new AbstractObjectCreationFactory() {
                        public Object createObject(Attributes attributes) throws Exception {
                            String name = attributes.getValue(Attribute.NAME);
                            String implClassName = attributes.getValue(Attribute.CLASS);

                            return model.createEventProviderHandler(name, implClassName);
                        }
                    });
            
            // association*
            digester.addFactoryCreate(
                    Path.EVENT_PROVIDER_ASSOCIATION, new AbstractObjectCreationFactory() {
                        public Object createObject(Attributes attributes) throws Exception {
//                            String supplierRole = attributes.getValue(Attribute.SUPPLIER_ROLE);
                            String clientRole = attributes.getValue(Attribute.CLIENT_ROLE);
                            String targetRef = attributes.getValue(Attribute.TARGET_REF);

                            EventProviderHandler ep = (EventProviderHandler)getDigester().peek();
                            StateMachine machine = resolveStateMachine(targetRef);

                            if (clientRole != null) {
                                return machine.createIncomingAssociation(ep, clientRole);
                            } else {
                                return machine.createIncomingAssociation(ep);
                            }
                        }
                    });
            
            // rootStateMachine
            digester.addFactoryCreate(
                    Path.ROOT_STATE_MACHINE, new AbstractObjectCreationFactory() {
                        public Object createObject(Attributes attributes) throws Exception {
                            String name = attributes.getValue(Attribute.NAME);
                            return resolveStateMachine(name);
                        }
                    });
            digester.addSetNext(
                    Path.ROOT_STATE_MACHINE, "setRootStateMachine");

            // stateMachine+
            digester.addFactoryCreate(
                    Path.STATE_MACHINE, new AbstractObjectCreationFactory() {
                        public Object createObject(Attributes attributes) throws Exception {
                            String name = attributes.getValue(Attribute.NAME);
                            currentMachine = createStateMachine(name);
                            
                            return currentMachine;
                        }
                    });

            // configStore
            digester.addFactoryCreate(
                    Path.CONFIG_STORE, new AbstractObjectCreationFactory() {
                        public Object createObject(Attributes attributes) throws Exception {
                            String className = attributes.getValue(Attribute.CLASS);
                            return className;
                        }
                    });
            digester.addSetNext(
                    Path.CONFIG_STORE, "setConfigManagerClassName");

            // association*
            digester.addFactoryCreate(
                    Path.STATE_MACHINE_ASSOCIATION, new AbstractObjectCreationFactory() {
                        public Object createObject(Attributes attributes) throws Exception {
                            String supplierRole = attributes.getValue(Attribute.SUPPLIER_ROLE);
//                            String clientRole = attributes.getValue(Attribute.CLIENT_ROLE);
                            String targetRef = attributes.getValue(Attribute.TARGET_REF);

                            StateMachine machine = (StateMachine) getDigester().peek();
                            ClassElement object = resolveClassElement(targetRef);
                            
                            if (object == null) {
                                throw new TransformException("Unresolved ref to class element: " + targetRef);
                            }

                            if (supplierRole != null) {
                                return machine.createOutgoingAssociation(object, supplierRole);
                            } else {
                                return machine.createOutgoingAssociation(object);
                            }
                        }
                    });

            // state (top state)
            digester.addFactoryCreate(
                    Path.TOP_STATE, new AbstractObjectCreationFactory() {
                        public Object createObject(Attributes attributes) throws Exception {
                            String name = attributes.getValue(Attribute.NAME);

                            StateMachine stateMachine = (StateMachine) getDigester().peek();
                            State top = stateMachine.createTopState(name);
                            stateMachine.setTop(top);
                            return top;
                        }
                    });

            // state*
            digester.addFactoryCreate(
                    Path.SUB_STATE, new AbstractObjectCreationFactory() {
                        public Object createObject(Attributes attributes) throws Exception {
                            String name = attributes.getValue(Attribute.NAME);
                            StateType type = StateType.parseStateType(
                                    attributes.getValue(Attribute.TYPE));

                            State superstate = (State) getDigester().peek();
                            State state = currentMachine.createState(name, type);
                            state.setSuperstate(superstate);
                            return state;
                        }
                    });

            // stateMachineRef*
            digester.addFactoryCreate(
                    Path.SUBMACHINE, new AbstractObjectCreationFactory() {
                        public Object createObject(Attributes attributes) throws Exception {
                            String name = attributes.getValue(Attribute.NAME);
                            return resolveStateMachine(name);
                        }
                    });
            digester.addSetNext(
                    Path.SUBMACHINE, "addSubmachine");

            // state/onEnterAction*
            digester.addFactoryCreate(
                    Path.ON_ENTER_ACTION, new AbstractObjectCreationFactory() {
                        public Object createObject(Attributes attributes) throws Exception {
                            String ident = attributes.getValue(Attribute.IDENT);

                            return currentMachine.createAction(ident);
                        }
                    });
            digester.addSetNext(
                    Path.ON_ENTER_ACTION, "addOnEnterAction");

            // transition*
            digester.addFactoryCreate(
                    Path.TRANSITION, new AbstractObjectCreationFactory() {
                        public Object createObject(Attributes attributes) throws Exception {
                            String eventName = attributes.getValue(Attribute.EVENT);
                            String expr = attributes.getValue(Attribute.GUARD);
                            String name = attributes.getValue(Attribute.NAME);

                            State source = currentMachine.findState(attributes.getValue(Attribute.SOURCE_REF));
                            State target = currentMachine.findState(attributes.getValue(Attribute.TARGET_REF));
                            Event event = eventName == null ? null : new Event(eventName);
                            Guard guard = currentMachine.createGuard(expr);

                            Transition transition = currentMachine.createTransition(source, target, guard, event);
                            transition.setName(name);
                            return transition;
                        }
                    });

            // outputAction*
            digester.addFactoryCreate(
                    Path.OUTPUT_ACTION, new AbstractObjectCreationFactory() {
                        public Object createObject(Attributes attributes) throws Exception {
                            String ident = attributes.getValue(Attribute.IDENT);

                            return currentMachine.createAction(ident);
                        }
                    });
            digester.addSetNext(
                    Path.OUTPUT_ACTION, "addOutputAction");


            Model model = (Model) digester.parse(xml);
            
            if (!unresolvedMachines.isEmpty()) {
                throw new TransformException("There are unresolved state machine refs: " + getUnreslvedStateMachinesList());
            }
            
            return model;
        } catch (TransformException e) {
            e.printStackTrace();
            throw e;
        } catch (Exception e) {
            e.printStackTrace();
            throw new TransformException(e);
        }
    }

    private String getUnreslvedStateMachinesList() {
        StringBuffer s = new StringBuffer("");
        for (Iterator i = unresolvedMachines.keySet().iterator(); i.hasNext();) {
            s.append(i.next()).append(" ");
        }
        
        return s.toString();
    }
    
    /* (non-Javadoc)
     * @see org.xml.sax.ErrorHandler#error(org.xml.sax.SAXParseException)
     */
    public void error(SAXParseException exception) throws SAXException {
        throw exception;
    }
    
    /* (non-Javadoc)
     * @see org.xml.sax.ErrorHandler#fatalError(org.xml.sax.SAXParseException)
     */
    public void fatalError(SAXParseException exception) throws SAXException {
        throw exception;
    }
    
    /* (non-Javadoc)
     * @see org.xml.sax.ErrorHandler#warning(org.xml.sax.SAXParseException)
     */
    public void warning(SAXParseException exception) throws SAXException {
        throw exception;
    }
    
    protected Model createModel(String name) {
        return Model.createModel(name);
    }
    
    protected ClassElement resolveClassElement(String name) {
        ClassElement ce = model.getControlledObjectHandler(name);
        
        if (ce == null) {
            ce = resolveStateMachine(name);
        }
        
        return ce;
    }

    protected StateMachine resolveStateMachine(String name) {
        StateMachine machine = model.getStateMachine(name);
        if (machine == null) {
            machine = (StateMachine) unresolvedMachines.get(name);
            if (machine == null) {
                machine = model.createStateMachine(name);
                unresolvedMachines.put(machine.getName(), machine);
            }
        }
        return machine;
    }

    protected StateMachine createStateMachine(String name) {
        StateMachine machine = (StateMachine) unresolvedMachines.get(name);
        if (machine != null) {
            unresolvedMachines.remove(machine.getName());
        } else {
            machine = model.createStateMachine(name);
        }
        return machine;
    }
    
    /**
     * Build in-memory model from given xml file.
     * Do not forget to compile model before use: 
     * <pre>
     *   StateMachineCompiler c = new StateMachineCompiler(null);
     *   DefaultCompilationListener cl = new DefaultCompilationListener();
     *   c.addCompilationListener(cl);
     *   c.compileWithIncluded(model.getRootStateMachine());
     * 
     *   String[] errors = cl.getErrors();
     *   if (errors.length > 0) {
     *       for (int i = 0; i < errors.length; i++) {
     *           System.out.println(errors[i]);
     *       }
     * 
     *       return;
     *   }
     * </pre>
     * 
     * @param is
     * @return
     * @throws TransformException
     */
    public static Model load(InputStream is) throws TransformException{
        return load(is, null);
    }
    
    /**
     * Read model from given {@link URL} and compiles it.
     * 
     * @param is input stream with automata model
     * @param xmlReader explicit XMLReader
     * @return read and compiled model
     * @throws IOException if an I/O exception occurs.
     * @throws TransformException if given resource couldn't be transformed
     */
    public static Model loadAndCompile(InputStream is, XMLReader xmlReader) throws IOException, TransformException {
        Model model = XMLToModel.load(is, xmlReader);

        /* compile state machine */
        StateMachineCompiler c = new StateMachineCompiler(null);
        DefaultCompilationListener cl = new DefaultCompilationListener();
        c.addCompilationListener(cl);
        
        for (Iterator i = model.getStateMachines().iterator(); i.hasNext(); ) {
            c.compile((StateMachine)i.next());
        }

        /* check compilation errors */
        String[] errors = cl.getErrors();
        if (errors.length > 0) {
            for (int i = 0; i < errors.length; i++) {
                l.fatal(errors[i]);
            }

            throw new TransformException("Model compilation error.");
        }
        
        return model;
    }
    
    /**
     * Read model from given {@link URL} and compiles it.
     * @param is input stream with automata model
     * @return read and compiled model
     * @throws IOException if an I/O exception occurs.
     * @throws TransformException if given resource couldn't be transformed
     */
    public static Model loadAndCompile(InputStream is) throws IOException, TransformException {
        return loadAndCompile(is, null);
    }

    /**
     * Build in-memory model from given xml file. Do not forget to compile model
     * before use:
     * 
     * <pre>
     * StateMachineCompiler c = new StateMachineCompiler(null);
     * DefaultCompilationListener cl = new DefaultCompilationListener();
     * c.addCompilationListener(cl);
     * c.compileWithIncluded(model.getRootStateMachine());
     * 
     * String[] errors = cl.getErrors();
     * if (errors.length &gt; 0) {
     *     for (int i = 0; i &lt; errors.length; i++) {
     *         System.out.println(errors[i]);
     *     }
     * 
     *     return;
     * }
     * </pre>
     * 
     * @param is
     * @return
     * @throws TransformException
     */
    public static Model load(InputStream is, XMLReader xmlReader) throws TransformException{
        XMLToModel x2m = create();

        return x2m.transform(is, xmlReader);
    }
}