/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.validation;

import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import antlr.RecognitionException;

import com.evelopers.unimod.analysis.FormulaResolver;
import com.evelopers.unimod.core.ModelElement;
import com.evelopers.unimod.core.ModelVisitor;
import com.evelopers.unimod.core.VisitorException;
import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.core.stateworks.Guard;
import com.evelopers.unimod.core.stateworks.State;
import com.evelopers.unimod.core.stateworks.StateMachine;
import com.evelopers.unimod.core.stateworks.Transition;

/**
 * Visits every state and checks completenes of guard condition set.
 *
 * @author Maxim Mazin
 * @version Revision: 1
 */
abstract class CompletenessCheckVisitor implements ModelVisitor {
    private StateMachine sm;

    public void setSm(StateMachine sm) {
        this.sm = sm;
    }
   
    public boolean visit(ModelElement e) throws VisitorException {
        if (e instanceof State) {
            State state = (State) e;
            if (! state.isComposite()) {
                checkState(state);
                return false;
            } else {
                return true;
            }
        }
        throw new VisitorException(e + " is not a state!");
    }

    protected void checkState(State state) throws VisitorException {
        StateTransitionsDecorator std = new StateTransitionsDecorator(state);
        
        for (Iterator i = std.getHandledEvents().iterator(); i.hasNext();) {
            Event event = (Event) i.next();

            /* else transition for given event makes condiitons system complete automatically */
            if (std.getElseTransition(event) == null) {
                checkCompleteness(state, event, std.getAllNotElseTransitions(event));
            }
        }
    }

    private void checkCompleteness(State state, Event event, List transitions) throws VisitorException {
        try {
        	/* Look for transitions with noncompiled guards */
        	boolean everyGuardIsCompiled = true;
        	for (Iterator i = transitions.iterator(); i.hasNext();) {
				Transition transition = (Transition) i.next();
				if (transition.getGuard().getAST() == null) {
					everyGuardIsCompiled = false;
					break;
				}
			}
        	
        	/* Don't check completeness until every guard will be compiled */
        	if (everyGuardIsCompiled) {
	            Guard residue = getResidue(transitions);
	            if (residue != null) {
	                fireIncompleteGuardSet(sm, state, event, residue);
	            }
        	}
        } catch (RecognitionException e) {
            throw new VisitorException(e);
        }
    }

    private Guard getResidue(List transitions) throws RecognitionException {
        Set guards = new HashSet();
        for (Iterator i = transitions.iterator(); i.hasNext();) {
            Transition transition = (Transition) i.next();
            guards.add(transition.getGuard().getAST());
        }
        FormulaResolver disjunction = FormulaResolver.createDisjunction(guards);
        if (disjunction.isTautology()) {
            return null;
        } else {
            return sm.createGuard(disjunction.getSimplifiedDNF(false));
        }      
    }

    abstract protected void fireIncompleteGuardSet(StateMachine sm, State state, Event event, Guard guard);
}
