/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.validation;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import com.evelopers.unimod.core.VisitorException;
import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.core.stateworks.Guard;
import com.evelopers.unimod.core.stateworks.State;
import com.evelopers.unimod.core.stateworks.StateMachine;

/**
 * Validates completeness (every state has complete system of transitions for every
 * event).
 * @author Maxim Mazin
 * @version Revision: 1
 */
class CompletenessValidator extends CompletenessCheckVisitor {
    private List completenessListeners = new ArrayList();

    /**
     * Adds <code>listener</code> to list of completeness listeners.
     * @param listener listener to add
     * @return true (as List).
     * @see CompletenessListener
     */
    public boolean addCompletenessListener(CompletenessListener listener) {
        return completenessListeners.add(listener);
    }

    /**
     * Removes <code>listener</code> from list of completeness listeners.
     * @param listener listener to remove.
     * @return true if list of listeners contained <code>listener</code>.
     * @see CompletenessListener
     */
    public boolean removeCompletenessListener(CompletenessListener listener) {
        return completenessListeners.remove(listener);
    }

    /**
     * Checks completenes of transitions for set that is {<code>state</code>} joined
     * with its descendants minus composite states. State is complete when
     * state has complete guard conditions set for every event, i.e.
     * for every combination of input actions values there is a transition. Every
     * completeness listener will be notified about every incomplete system of
     * transitions.
     * @param sm <code>StateMachine</code> to validate.
     */
    public void validateCompleteness(StateMachine sm) {
        if (sm.getTop() == null) {
            throw new IllegalArgumentException("State machine "+sm.getName()+
                    " doesn't have top state");
        }
        setSm(sm);
        try {
            sm.getTop().accept(this);
        } catch (VisitorException e) {
            e.printStackTrace();
        }
    }

    /**
     * Check completeness for single state.
     * @param state non composite state
     */
    public void validateCompleteness(StateMachine sm, State state) {
        setSm(sm);
        try {
            if (! state.isComposite()) {
                checkState(state);
            }
        } catch (VisitorException e) {
            e.printStackTrace();
        }
    }

    protected void fireIncompleteGuardSet(StateMachine sm, State state, Event event, Guard guard) {
        for (Iterator i = completenessListeners.iterator(); i.hasNext();) {
            CompletenessListener completenessListener = (CompletenessListener) i.next();
            completenessListener.incompleteGuardSet(sm, state, event, guard);
        }
    }
}