/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.validation;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import com.evelopers.unimod.core.VisitorException;
import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.core.stateworks.Guard;
import com.evelopers.unimod.core.stateworks.State;
import com.evelopers.unimod.core.stateworks.StateMachine;
import com.evelopers.unimod.core.stateworks.Transition;

/**
 * Validates consistency (every state has consistent system of transitions for every
 * event).
 * @author Maxim Mazin
 * @version Revision: 1
 */
class ConsistencyValidator extends ConsistencyCheckVisitor {
    private List consistencyListeners = new ArrayList();

    /**
     * Adds <code>listener</code> to list of consistency listeners.
     * @param listener listener to add
     * @return true (as List).
     * @see ConsistencyListener
     */
    public boolean addConsistencyListener(ConsistencyListener listener) {
        return consistencyListeners.add(listener);
    }

    /**
     * Removes <code>listener</code> from list of consistency listeners.
     * @param listener listener to remove.
     * @return true if list of listeners contained <code>listener</code>.
     * @see ConsistencyListener
     */
    public boolean removeConsistencyListener(ConsistencyListener listener) {
        return consistencyListeners.remove(listener);
    }

    /**
     * Checks if every state has consistent pairs of transitions, i.e. any combination
     * of input actions and event will meet guard condition only on the one transition.
     * Will be checked all ancestors and all descendants of the <code>state</code>
     * @param sm <code>StateMachine</code> to validate.
     */
    public void validateConsistency(StateMachine sm) {
        if (sm.getTop() == null) {
            throw new IllegalArgumentException("State machine "+sm.getName()+
                    " doesn't have top state");
        }
        setSm(sm);
        try {
            /* Check descendants' consistency */
            sm.getTop().accept(this);
        } catch (VisitorException e) {
            e.printStackTrace();
        }
    }

    public void validateConsistency(StateMachine sm, State state) {
        setSm(sm);
        try {
            checkState(state);
        } catch (VisitorException e) {
            e.printStackTrace();
        }
    }

    protected void fireInconsistentGuardSet(StateMachine sm, Transition t1, Transition t2, Guard guard) {
        for (Iterator i = consistencyListeners.iterator(); i.hasNext();) {
            ConsistencyListener consistencyListener = (ConsistencyListener) i.next();
            consistencyListener.inconsistentGuardSet(sm, t1, t2, guard);
        }
    }

    protected void fireRedundantElseTransitions(StateMachine sm, State state, Event e, int elseTransitionsNumber) {
        for (Iterator i = consistencyListeners.iterator(); i.hasNext();) {
            ConsistencyListener consistencyListener = (ConsistencyListener) i.next();
            consistencyListener.redundantElseTransitions(sm, state, e, elseTransitionsNumber);
        }
    }
}