/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.validation;

import java.util.Collection;

import com.evelopers.unimod.core.stateworks.State;
import com.evelopers.unimod.core.stateworks.StateMachine;

/**
 * <p>
 * State machine structure validator.
 * </p>
 * <p>
 * Implements the following validation rules:
 * <li> 1. Structure. Only one INITIAL state must present in every attainable composite state.
 * {@link StructureListener}, {@link #validateStructure()};
 * <li> 2. Attainability. Path from the most upper INITAL state to every state must present.
 * {@link AttainabilityListener}, {@link #validateAttainability()};
 * <li> 3. Consistency. {@link ConsistencyListener}, {@link #validateConsistency};
 * <li> 4. Completeness. {@link CompletenessListener}, {@link #validateCompleteness};
 * </p>
 *
 * @author Vadim Gurov, Maxim Mazin
 * @version $Revision: 1$
 */
public class StateMachineValidator {
	private StateMachine sm;

    private StructureValidator      structureValidator;
    private ConsistencyValidator    consistencyValidator;
    private CompletenessValidator   completenessValidator;

	public static StateMachineValidator createValidator(StateMachine sm) {
		return new StateMachineValidator(sm);
	}

    protected StateMachineValidator(StateMachine sm) {
        this.sm = sm;

        structureValidator      = new StructureValidator();
        consistencyValidator    = new ConsistencyValidator();
        completenessValidator   = new CompletenessValidator();
    }

    /* Event listeners */

    public boolean addStructureListener(StructureListener listener) {
        return structureValidator.addStrucutreListener(listener);
    }

    public boolean removeStructureListener(StructureListener listener) {
        return structureValidator.removeStrucutreListener(listener);
    }

    public boolean addConsistencyListener(ConsistencyListener listener) {
        return consistencyValidator.addConsistencyListener(listener);
    }

    public boolean removeConsistencyListener(ConsistencyListener listener) {
        return consistencyValidator.addConsistencyListener(listener);
    }

    public boolean addCompletenessListener(CompletenessListener listener) {
        return completenessValidator.addCompletenessListener(listener);
    }

    public boolean removeCompletenessListener(CompletenessListener listener) {
        return completenessValidator.addCompletenessListener(listener);
    }


    /* Validation actions */

    /**
     * Validates structure.
     * @see com.evelopers.unimod.validation.StructureValidator#validateStructure
     */
    public void validateStructure() {
        structureValidator.validateStructure(sm);
    }

    /**
     * Validates consistency of top-state.
     * @see #validateConsistency(State)
     */
    public void validateConsistency() {
        consistencyValidator.validateConsistency(sm);
    }

    /**
     * Validates consistency of <code>state</code> its descendants and ancestors.
     * @param state state consistency of what should be checked.
     * @see com.evelopers.unimod.validation.ConsistencyValidator#validateConsistency(com.evelopers.unimod.core.stateworks.StateMachine, com.evelopers.unimod.core.stateworks.State)
     */
    public void validateConsistency(State state) {
        consistencyValidator.validateConsistency(sm, state);
    }

    /**
     * Validate completeness of top-state.
     * @see #validateCompleteness(State)
     */
    public void validateCompleteness() {
        completenessValidator.validateCompleteness(sm);
    }

    /**
     * Validates completeness of state and its lief-descedants os <code>state</code>.
     * Lief-descendants are descendents of <code>state</code> which are not composite.
     * <b>Note:</b> completeness of <code>state</code> will be checked only if
     * <code>state</code> isn't composite.
     * @param state state completeness of what should be checked.
     * @see com.evelopers.unimod.validation.CompletenessValidator#validateCompleteness(com.evelopers.unimod.core.stateworks.StateMachine, com.evelopers.unimod.core.stateworks.State)
     */
    public void validateCompleteness(State state) {
        completenessValidator.validateCompleteness(sm, state);
    }


    /* Sets getters */
    public Collection getAttainableStates() {
        return structureValidator.getAttainableStates();
    }
    
    /* Convinience */

    protected StateMachine getStateMachine() {
        return sm;
    }
}

