/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.validation;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.evelopers.unimod.core.stateworks.State;
import com.evelopers.unimod.core.stateworks.StateMachine;
import com.evelopers.unimod.core.stateworks.StateType;
import com.evelopers.unimod.core.stateworks.Transition;

/**
 * Builds map of attainable states for statemachines. Concrete statemachine
 * is a key in this map and set of attainable states is a value.
 *
 * @author Maxim Mazin
 * @version Revision: 1
 */
class StructureVisitor {
    /** Set of attainable states */
    protected Set attainableStates;

    /** Set of attainable composite states which have no initial states */
    protected Set initialStateLess;

    protected Map redundantInitStates;
    
    /** Set of not root state machines that has associated event providers */
    protected Set notRootMachinesWithEventProviders;

    /**
     * Constructs visitor
     */
    public StructureVisitor() {
        attainableStates = new HashSet();
        initialStateLess = new HashSet();
        redundantInitStates = new HashMap();
        notRootMachinesWithEventProviders = new HashSet();
    }

    /**
     * Visits states and builds set of attainable states.
     * @param state state to visit.
     */
    public void visit(State state) {
        if (! alreadyVisited(state)) {
            attainableStates.add(state);
            visitIncludedStateMachines(state);
            visitInitialState(state);
            followTransitions(state);
        }
    }
    
    protected void visitIncludedStateMachines(State state) {
        for (Iterator i = state.getSubmachines().iterator(); i.hasNext(); ) {
            StateMachine sm = (StateMachine)i.next(); 
            List ep = sm.getEventProviderHandlers();
            
            if (ep.size() > 0) {
                notRootMachinesWithEventProviders.add(sm);
            }
        }
    }
    
    public Set getNotRootMachinesWithEventProviders() {
        return notRootMachinesWithEventProviders;
    }
    
    /**
     * @return Set of attainable states.
     */
    public Set getAttainableStates() {
        return attainableStates;
    }

    /**
     * Returns set of attainable composite states which have no initial states.
     * @return set of attainable composite states which have no initial states
     */
    public Set getNoInitState() {
        return initialStateLess;
    }

    /**
     * Returns map of attainable composite states which have more then one initial state.
     * @return map of attainable composite states which have more then one initial state
     */
    public Map getRedundantInitStates() {
        return redundantInitStates;
    }

    /**
     * If <code>state</code> is composite then visit its initial state. If composite
     * state has no initial state then memorize this fact.
     * @see #getNoInitState()
     */
    private void visitInitialState(State state) {
        if (state.isComposite()) {
            int initStateNumber = 0;
            for (Iterator i = state.getSubstates().iterator(); i.hasNext();) {
                State substate = (State) i.next();
                if (StateType.INITIAL.equals(substate.getType())) {
                    initStateNumber ++;
                    visit(substate);
                }
            }
            if (initStateNumber == 0) {
                initialStateLess.add(state);
            } else if (initStateNumber > 1) {
                redundantInitStates.put(state, new Integer(initStateNumber));
            }
        }
    }

    /**
     * Visits all states that are adjacent to the <code>state</code> or to the superstate
     * of the <code>state</code>.
     */
    private void followTransitions(State state) {
        if (state.hasSuperstate()) {
            followTransitions(state.getSuperstate());
        }

        List transitions = state.getOutgoingTransitions();
        for (Iterator i = transitions.iterator(); i.hasNext();) {
            visit(((Transition) i.next()).getTargetState());
        }
    }

    /**
     * Checks if <code>state</code> was already visited.
     * @return true if <code>state</code> is already visited.
     */
    private boolean alreadyVisited(State state) {
        return attainableStates.contains(state);
    }
}