/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.validation;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;

import javax.xml.parsers.ParserConfigurationException;

import org.xml.sax.SAXException;

import com.evelopers.unimod.compilation.DefaultCompilationListener;
import com.evelopers.unimod.compilation.StateMachineCompiler;
import com.evelopers.unimod.core.stateworks.StateMachine;
import com.evelopers.unimod.transform.TransformException;
import com.evelopers.unimod.transform.xml.XMLToModel;

/**
 * Validates <code>StateMachine</code> described by xml and dumps all errors.
 *
 * @author Maxim Mazin
 * @version Revision: 1
 */
public class ValidationTool {

    public static void main(String[] args) {
        if (args.length < 1) {
            System.out.println("Usage: java ValidationTool <StateMachine>.xml");
        } else {
            try {
                ValidationTool vt = new ValidationTool(new FileInputStream(args[0]));
                
                if (vt.compile()) {
                    vt.validate();
                }
                
            } catch (FileNotFoundException e) {
                System.err.println("File "+args[0]+" was not found");
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    private StateMachineValidator validator;
    private StateMachine stateMachine;    

    /**
     * Creates <code>ValidationTool</code> for validating <code>StateMachine</code>
     * whose xml description is found in <code>inputXML</code>.
     * @param inputXML <code>InputStream</code> with xml description of <code>StateMachine</code>.
     * @throws IOException if any I/O errors occur.
     * @throws SAXException if any xml-parse errors occur.
     * @throws TransformException if any transform error occur.
     * @throws ParserConfigurationException if any configuration error occur.
     */
    public ValidationTool(InputStream inputXML) throws
            IOException, SAXException, TransformException, ParserConfigurationException {
        stateMachine = transformXMLToStateMachine(inputXML);
        
        ValidationMessenger validationMessanger = new ValidationMessenger();
        validator = StateMachineValidator.createValidator(stateMachine);
        validator.addStructureListener(validationMessanger);
        validator.addConsistencyListener(validationMessanger);
        validator.addCompletenessListener(validationMessanger);
    }

    public boolean compile() {
        StateMachineCompiler smc = new StateMachineCompiler(null);
        DefaultCompilationListener dcl = new DefaultCompilationListener();
        smc.addCompilationListener(dcl);
        smc.compileWithIncluded(stateMachine);
        
        if (dcl.getErrors() != null && dcl.getErrors().length > 0) {
            System.out.println("State machine compilation errors:");
            for (int i = 0; i < dcl.getErrors().length; i++) {
            	System.out.println(dcl.getErrors()[i]);
            }
            System.out.println("Validation cancelled.");
            
            return false;
        }
        
        return true;
    }
    
    /**
     * @return Validator. It may be used to subscribe to events.
     */
    public StateMachineValidator getValidator() {
        return validator;
    }

    /**
     * Validate <code>StateMachine</code>.
     */
    public void validate() {
        validator.validateStructure();
        validator.validateConsistency();
        validator.validateCompleteness();
    }

    private StateMachine transformXMLToStateMachine(InputStream inputXML) throws
            IOException, SAXException, TransformException, ParserConfigurationException {
        XMLToModel transformer = XMLToModel.create();

        return (StateMachine) transformer.transform(inputXML).getRootStateMachine();
    }
}