/*
 * Developed by eVelopers Corporation - May 31, 2003
 *
 * Copyright (c) 1999-2003 eVelopers Corporation. All rights reserved.
 * This software is the confidential and proprietary information of
 * eVelopers Corporation. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms of
 * the license agreement you entered into with eVelopers.
 *
 * $Date: 26-Jun-03 11:29:30$
 *
 */
package com.evelopers.common.concurrent;

/**
 * Post box is an abstraction for synchronization purposes. Sender puts
 * messages to the post box and receiver gets them. If a receiver wants to
 * get a message and there is no messages in the post box the receiver is
 * blocked until a sender puts a message. Post box may be created with
 * deficit or proficit of messages. If a post box is created with proficit
 * a receiver can pick all of them up before it is blocked. If a post box is
 * created with deficit sender must put as much messages as the deficit value
 * plus 1 to allow receivers to get messages without blocking.
 *
 * @author danis
 * @version $Revision: 2$
 */
public class PostBox {

	private MsgCount sent;

	/**
	 * Creates an empty post box. If {@link #await()} or {@link #receive()}
	 * is called earleir than {@link #send()} calling thread will be blocked.
	 */
	public PostBox() { this(0); }

	/**
	 * Creates a post box with specified initial filling. 0 means an empty
	 * post box. If a positive value is provided client can {@link #receive()}
	 * without blocking as many times as the specified value. If a negative
	 * value is provided client must {@link #send()} as many times as the
	 * specified value plus 1 before {@link #receive()} can run without
	 * blocking.
	 *
	 * @param initialFilling positive value prefills the post box with
	 * specified number of messages, while negative one vacuums the post box.
	 * 0 designates an empty post box. At least one message must be put to
	 * allow receiving client to continue running.
	 */
	public PostBox(long initialFilling) {
		this.sent = new MsgCount(initialFilling);
	}

	/**
	 * Return <code>true</code> if there is at least one message
	 * in the post box. Otherwise, returns <code>false</code>.
	 *
	 * @return <code>false</code> if the post box is either empty or vacuumed;
	 * otherwise, <code>false</code> is returned.
	 */
	public synchronized boolean available() {
		return (sent.count > 0);
	}

	/**
	 * Awaits for a message. If the post box is either empty or vacuumed a call
	 * to this method blocks execution. If there is some messages in the post
	 * box then the call returns without retrieving a message. This method
	 * does not change the number of messages.
	 *
	 * @throws java.lang.InterruptedException if blocked thread is interrupted
	 * by another.
	 */
	public synchronized void await() throws InterruptedException {
		while(!available()) wait();
	}

	/**
	 * Receives a message. If there is no messages in the post box a call to
	 * this method blocks execution. If the post box is not empty the method
	 * gets one message from the post box and returns.
	 *
	 * @throws java.lang.InterruptedException if blocked thread is interrupted
	 * by another.
	 */
	public synchronized void receive() throws InterruptedException {
		await();
		sent.count -= 1;
	}

	/**
	 * Adds a message to the post box. If there is some blocked threads
	 * awaiting for a message all of them are awaken and start to compete in
	 * receiving a message. Only one thread gets a message. All other threads
	 * fall into sleep until the post box is sent a message again.
	 */
	public synchronized void send() {
		sent.count += 1;
		notifyAll();
	}

	public String toString() {
		return getClass().getName() + "[" + sent.count + "]";
	}


	private static final class MsgCount {
		public long count;
		public MsgCount(long init) { this.count = init; }
		public String toString() {
			return getClass().getName() + "[" + count + "]";
		}
	}
} ///:~