/*
 * Developed by eVelopers Corporation
 *
 * Copyright (c) 1999-2003 eVelopers Corporation. All rights reserved.
 * This software is the confidential and proprietary information of
 * eVelopers Corporation. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms of
 * the license agreement you entered into with eVelopers.
 *
 * $Date: 28-Apr-04 16:04:02$
 *
 */

package com.evelopers.common.log.log4j;

import java.util.*;
import org.apache.log4j.*;
import org.apache.log4j.helpers.*;
import org.apache.log4j.spi.*;

/**
 * <p>This class is the extension of Log4J AsyncAppender class
 * (<code>org.apache.log4j.AsyncAppender</code>). Its main purpose is to add
 * ability to get more useful information when some errors occur and avoid
 * extensive logging in normal conditions.</p>
 * <p>RingBufferAsyncAppender uses internal message buffer to constantly collect
 * messages with low priority (usually DEBUG, see parameter threshold below),
 * which is dumped when error occurs (see parameter <code>bufferDumpThreshold</code>
 * below). The output in normal conditions is controlled by parameter
 * <code>passthroughThreshold</code>.</p>
 * <p>Configuration of RingBufferAsyncAppender (as well as original AsyncAppender)
 * is possible using XML form of configuration file ONLY (see description of
 * Log4J <code>AsyncAppender</code> for details).</p>
 * Here is the example of RingBufferAsyncAppender usage (extract of log4j.xml).
 *
 * <pre>
 * &nbsp;&lt;?xml version="1.0" encoding="UTF-8" ?&gt;
 *
 * &nbsp;&lt;!DOCTYPE log4j:configuration SYSTEM "log4j.dtd"&gt
 * &nbsp;&lt;log4j:configuration xmlns:log4j="http://jakarta.apache.org/log4j/"&gt
 * &nbsp;
 * &nbsp;  &lt;appender name="console" class="org.apache.log4j.ConsoleAppender"&gt
 * &nbsp;      &lt;layout class="org.apache.log4j.PatternLayout"&gt
 * &nbsp;          &lt;param name="ConversionPattern" value="[%d] %-5p [%x] %m (%c)%n"/&gt
 * &nbsp;      &lt;/layout&gt
 * &nbsp;  &lt;/appender&gt
 * &nbsp;
 * &nbsp;  &lt;appender name="RingBuffer" class="com.evelopers.common.log.log4j.RingBufferAsyncAppender"&gt
 * &nbsp;      &lt;param name="threshold"            value="DEBUG"/&gt
 * &nbsp;      &lt;param name="passthroughThreshold" value="DEBUG"/&gt
 * &nbsp;      &lt;param name="bufferDumpThreshold"  value="ERROR"/&gt
 * &nbsp;      &lt;param name="bufferSize"           value="50"/&gt
 * &nbsp;      &lt;appender-ref ref="console"/&gt
 * &nbsp;  &lt;/appender&gt
 * &nbsp;
 * &nbsp;  &lt;root&gt
 * &nbsp;      &lt;priority value ="DEBUG" /&gt
 * &nbsp;      &lt;appender-ref ref="RingBuffer" /&gt
 * &nbsp;  &lt;/root&gt
 * &nbsp;&lt;/log4j:configuration&gt
 * </pre>
 *
 * <p>As well as original AsyncAppender RingBufferAsyncAppender allows to add any
 * number of other appenders to do actual logging (using <code>&lt;appender-ref&gt;</code> tag).
 * Here is the description of RingBufferAsyncAppender parameters:</p>
 *
 * <ul>
 * <li><b>threshold</b>&nbsp;	The lowest priority level of log messages accepted
 * by this appender. This parameter is common for all Log4J appenders. All
 * messages with this and upper priority will be stored in appenders ring buffer.
 * Any allowed priority level.</li>
 * <br/>
 * <li><b>passthroughThreshold</b>&nbsp;	The lowest priority level of log messages,
 * which are passed to connected appenders. In other words only messages with
 * this priority will be logged by the loggers connected to this
 * RingBufferAsyncAppender.	Any allowed priority level. INFO by default (if omitted).</li>
 * <br/>
 * <li><b>bufferDumpThreshold</b>&nbsp;	The lowest priority level of log messages
 * caused dumping of buffered.	Any allowed priority level. ERROR by default (if omitted).</li>
 * <br/>
 * <li><b>bufferSize</b>&nbsp;	The size of ring buffer (number of messages).
 * Integer number. 50 by default (if omitted).</li>
 * </ul>
 * USAGE NOTES:
 * <ol>
 * <li>Usage of RingBufferAsyncAppender without connected general (synchronous)
 * appenders has no sense. RingBufferAsyncAppender doesnt do any really logging
 * itself.</li>
 * <br/>
 * <li>The value of threshold parameter should be DEBUG usually to log as much
 * information as possible in the ring buffer.</li>
 * <br/>
 * <li>The <code>passthroughThreshold</code> parameter controls the really level
 * of logging and should be changed as necessary, i.e. the log messages with
 * priority >= <code>passthroughThreshold</code> will be logged in normal conditions.</li>
 * <br/>
 * <li>The value of <code>bufferDumpThreshold</code> parameter shouldnt be greater
 * than ERROR usually.</li>
 * <br/>
 * <li> None of this class methods is intended to be invoked directly.
 * All of them are invoked by Log4J system. </li>
 * </ol>
 *
 * Author: L.V.A
 * @version $Revision: 1$
 */
public class RingBufferAsyncAppender extends AsyncAppender {

  private int bufferSize = 0;
  private RingBuffer buffer = null;
  private Level bufferDumpThreshold = null;
  private Level passthroughThreshold = null;

  /**
   * Public constructor (invoked by Log4J configurator)
   *
   */
  public RingBufferAsyncAppender() {

    bufferSize = 100;
    bufferDumpThreshold = Level.ERROR;
    passthroughThreshold = Level.INFO;
  }

  /**
   * This method is invoked by Log4J configurator to apply read configuration
   * parameters.
   *
   */
  public void activateOptions() {

    if (bufferSize > 0)
      buffer = new RingBuffer(bufferSize);
    else
      LogLog.warn(
        "The size of temporary buffer has not been set. Appender is disabled.");
  }

  /**
   * Set the <code>bufferSize</code> parameter (invoked by Log4J configurator).
   *
   * @param <code>bufferSize</code> new parameter's value
   */
  public void setBufferSize(int bufferSize) {
    this.bufferSize = bufferSize;
  }

  /**
   * Set the <code>bufferDumpThreshold</code> parameter
   * (invoked by Log4J configurator).
   *
   * @param <code>bufferDumpThreshold</code> new parameter's value
   */
  public void setBufferDumpThreshold(String bufferDumpThreshold) {
    this.bufferDumpThreshold = Level.toLevel(bufferDumpThreshold);
  }

  /**
   * Set the <code>setPassthroughThreshold</code> parameter
   * (invoked by Log4J configurator).
   *
   * @param <code>passthroughThreshold</code> new parameter's value
   */
  public void setPassthroughThreshold(String passthroughThreshold) {
    this.passthroughThreshold = Level.toLevel(passthroughThreshold);
  }

  /**
   * This method overrides <code>append</code> method of <code>AsyncAppender</code>
   * and implements basic logging logic described above.
   *
   * @param <code>loggingEvent</code>
   */
  public void append(LoggingEvent loggingEvent) {

    Level level = loggingEvent.getLevel();

    if (level.isGreaterOrEqual(passthroughThreshold))
      super.append(loggingEvent);
    if (!getThreshold().isGreaterOrEqual(passthroughThreshold)) {
      if (level.isGreaterOrEqual(bufferDumpThreshold))
        dumpBuffer();
      else
        buffer.append(loggingEvent);
    }
  }

  private final void dumpBuffer() {

    super.append(
      new LoggingEvent(
        "", Category.getInstance(getClass().getName()), bufferDumpThreshold,
        "============= START OF TEMPORARY BUFFER DUMP ============", null));
    List list = buffer.toList();

    for (Iterator iterator = list.iterator(); iterator.hasNext();
         super.append((LoggingEvent)iterator.next()));
    super.append(
      new LoggingEvent(
        "", Category.getInstance(getClass().getName()), bufferDumpThreshold,
        "============= END OF TEMPORARY BUFFER DUMP ============", null));
  }

  // =================== Simple Ring Bufer implementation ====================
  private class RingBuffer {

    private Object buffer[] = null;
    private int currentPointer = 0;
    private int lastPointer = 0;

    public synchronized void append(Object obj) {

      lastPointer = currentPointer;
      buffer[currentPointer++] = obj;
      if (currentPointer >= buffer.length)
        currentPointer = 0;
    }

    public synchronized List toList() {

      ArrayList arraylist = new ArrayList();
      int i;

      for (i = currentPointer; i != lastPointer; ) {
        if (buffer[i] != null)
          arraylist.add(buffer[i]);
        if (++i >= buffer.length)
          i = 0;
      }

      if (buffer[i] != null)
        arraylist.add(buffer[i]);
      return arraylist;
    }

    public RingBuffer(int i) {

      currentPointer = 0;
      lastPointer = 0;
      buffer = new Object[i];
    }
  }

}
