/*
 * Developed by eVelopers Corporation
 *
 * Copyright (c) 1999-2002 eVelopers Corporation. All rights reserved.
 * This software is the confidential and proprietary information of
 * eVelopers Corporation. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms of
 * the license agreement you entered into with eVelopers.
 *
 * $Date: 19-Jul-05 13:23:14$
 *
 */
package com.evelopers.common.util.csv;

import java.io.*;
import com.evelopers.common.util.helper.*;
import com.evelopers.common.exception.*;

/**
 * <p>
 * Utility class for encoding comma separated values (CSV) streams.
 * </p>
 * <p>
 * Has inner interface DataModel that represents call back interface to
 * data meta and not meta information to be encoded into CSV compatible stream.
 * Generally, input data are presented as table with columns and rows. Columns may
 * have names - it's some kind of meta data.
 * </p>
 *
 * @author magus
 * @author kex
 * @version $Revision: 6$
 */
public class CSVEncoder {

  /**
   * Interface to be implemented by consumer of CSV export service to
   * provide interface to data meta model and data itself to be exported.
   */
  public interface DataModel {

    /**
     * Indicate that during export no need to flush output stream.
     */
    public static final int NO_FLUSH = -1;

    /**
     * Must returns value of given column from given row.
     * @param column column index starting with 0 and ending with {@link #getColumnCount}-1
     * @param row row index starting with 0 and ending with {@link #getRowCount}-1
     * @return value for given column and row
     * @throws CommonException this exception will be thrown by originally called
     * <code>export</code> method back to caller
     */
    public Object get(int column, int row) throws CommonException;

    /**
     * Must return total amount of columns
     * @return total amount of columns
     * @throws CommonException this exception will be thrown by originally called
     * <code>export</code> method back to caller
     */
    public int getColumnCount() throws CommonException;

    /**
    * Must return total amount of rows
    * @return total amount of rows
    * @throws CommonException this exception will be thrown by originally called
     * <code>export</code> method back to caller
    */
    public int getRowCount() throws CommonException;

    /**
     * Must return true if header with column names has to be added, false - otherwise.
     * @return true if header with column names must be added, false - otherwise
     * @throws CommonException this exception will be thrown by originally called
     * <code>export</code> method back to caller
     */
    public boolean needHeader() throws CommonException;

    /**
     * Must return name for given column index
     * @param column column index starting with 0 and ending with {@link #getColumnCount}-1
     * @return column name
     * @throws CommonException this exception will be thrown by originally called
     * <code>export</code> method back to caller
     */
    public String getColumnName(int column) throws CommonException;

    /**
     * Must return amount of rows after which data will be writen to the output stream
     * and output stream will be flushed.
     * Must return {@link #NO_FLUSH} if no flush needed at all.
     * @return amount of rows after with output stream will be flushed.
     * @throws CommonException this exception will be thrown by originally called
     * <code>export</code> method back to caller
     */
    public int getFlushCount() throws CommonException;
  }

  /**
   * Private constructor deny to create instcance of this class.
   */
  private CSVEncoder() {
  }

  /**
   * Wraper for {@link #export(CSVEncoder.DataModel, PrintWriter)} method.
   * @param dm DataModel to export
   * @param os output stream
   * @throws IOException thrown by output streams write methods
   * @throws CommonException CommonException if columns <= 0 or rows < 0 as
   * {@link com.evelopers.common.exception.ApplicationException}. Also thrown by
   * methods of given data model.
   * @see #export(CSVEncoder.DataModel, PrintWriter)
   */
  public static void export(DataModel dm, OutputStream os) throws IOException, CommonException {
    export(dm, new PrintWriter(os));
  }

  /**
   * Exports given data model into the given print writer in CSV compatible format.
   * Closes given print writer at the end of export.
   * @param dm data model that describes data to export
   * @param pw print writer to export data to
   * @throws IOException thrown by output streams write methods
   * @throws CommonException if columns <= 0 or rows < 0 as
   * {@link com.evelopers.common.exception.ApplicationException}. Also thrown by
   * methods of given data model.
   */
  public static void export(DataModel dm, PrintWriter pw) throws IOException, CommonException {
    StringBuffer result = new StringBuffer();

    int columns = dm.getColumnCount();
    int rows = dm.getRowCount();
    int flush = dm.getFlushCount();

    if (columns <= 0 || rows < 0) {
      throw new LogicException("Incorrect value of coulmn count [" + columns + "] or row count [" + rows + "]");
    }

    // append header if needed
    if (dm.needHeader()) {
      for (int i = 0; i < columns - 1; i++) {
         result.append(getCSVString(dm.getColumnName(i))).append(CSVDecoder.SCSV_DELIMETER);
      }

      result.append(getCSVString(dm.getColumnName(columns - 1))).append("\n");
    }

    int f = 0;
    for (int r = 0; r < rows; r++, f++) {
      for (int c = 0; c < columns - 1; c++) {
        result.append(getCSVString(dm.get(c, r))).append(CSVDecoder.SCSV_DELIMETER);
      }

      result.append(getCSVString(dm.get(columns - 1, r))).append("\n");

      if (f >= flush && flush != DataModel.NO_FLUSH) {
        pw.print(result.toString());
        pw.flush();

        f = 0;
        result = new StringBuffer();
      }
    }

    pw.print(result.toString());
    pw.flush();    
  }

  /**
   * Returns CSV compatible string form.
   * Replaces all double quoters with triple double quoters.
   * Add leading and final double quoters if original string contains comma.
   *
   * @param str string to convert to CSV compatible form
   * @return CSV compatible string
   */
  public static String getCSVString(String str) {

      if (str == null) {
          return "";
      }

      str = StringHelper.replaceAll(str, "\"", "\"\"");

      str = "\"" + str + "\"";

      return str;
  }

  /**
   * Returns CSV compatible string form of given object.
   *
   * @param o object to convert to CSV compatible form
   * @return CSV compatible string
   */
  public static String getCSVString(Object o) {
      return getCSVString(o == null ? null : o.toString());
  }

}
