/*
 * Developed by eVelopers Corporation
 *
 * Copyright (c) 1999-2003 eVelopers Corporation. All rights reserved.
 * This software is the confidential and proprietary information of
 * eVelopers Corporation. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms of
 * the license agreement you entered into with eVelopers.
 *
 * $Date: 08-Apr-04 18:00:57$
 *
 */
package com.evelopers.common.util.helper;

import com.evelopers.common.exception.*;

import java.lang.reflect.*;
import java.net.URL;
import java.text.MessageFormat;
import java.util.StringTokenizer;

/**
 * Classes instaciator helper. Implements class loading and object instantiate
 * methods. Also implements method to call method on some object with reflection.
 *
 * @author magus
 * @author kex
 * @version $Revision: 14$
 */
public class ClassHelper {

    /**
     * Private constructor deny to create instcance of this class.
     */
    private ClassHelper() {
    }

    /**
     * Loads class with given name. First try to use current thread context
     * class loader, if fails - try to use class it's own class loader, if fails -
     * try to use system class loader.
     *
     * @param className class name to load
     * @return newly loaded class
     * @throws SystemException if can't load class. Exception wraps ClassNotFoundException.
     * @see java.lang.Class#getClassLoader()
     * @see java.lang.ClassLoader
     * @see java.lang.ClassLoader#getSystemClassLoader()
     * @see java.lang.Thread#currentThread()
     * @see java.lang.Thread#getContextClassLoader()
     */
    public static final Class loadClass(String className) throws SystemException {
        try {
            ClassLoader tcl = Thread.currentThread().getContextClassLoader();
            Class oClass = null;
            if (tcl != null) {
                try {
                    oClass = tcl.loadClass(className);
                } catch (ClassNotFoundException e) {
                }
            }

            if (oClass == null) {
                ClassLoader cl = ClassHelper.class.getClassLoader();
                if (cl != null) {
                    try {
                        oClass = cl.loadClass(className);
                    } catch (ClassNotFoundException e) {
                    }
                }
            }

            if (oClass == null) {
                oClass = ClassLoader.getSystemClassLoader().loadClass(className);
            }

            return oClass;

        } catch (ClassNotFoundException e) {
            throw new SystemException(e, "Can't load class [" + className + "]");
        }
    }

    
    /**
     * Instanciates class using appropriate constructor and checks if it is of correct class
     *
     * @param cName class name to instantiate
     * @param paramClasses class constructor parameters' classes
     * @param params class constructor parameters
     * @param clazz class of class to instantiate
     * @return new object of given class
     * @throws SystemException if some errors occur during object creating such as
     *         absence of constructor with given parameters set
     * @throws LogicException if new object is not of clazz class
     */
    public static final Object instantiate(String cName,
                                           Class[] paramClasses,
                                           Object[] params,
                                           Class clazz) throws LogicException, SystemException {
        Object obj = instantiate(cName, paramClasses, params);
        if (!clazz.isInstance(obj)) {
            throw new LogicException("Class " + cName + " is of incorrect class");
        }
        return obj;
    }
    
    /**
     * Instanciates class using appropriate constructor. Loads class using
     * {@link #loadClass(String)} method.
     * Uses specified array of classes to find the appropriate constructor
     * and transferred specified parameters to constructor for an instance
     * creation.
     *
     * @param cName class name to instantiate
     * @param paramClasses class constructor parameters' classes. If null then tries
     * to extract classes from params array
     * @param params class constructor parameters. If null then creates
     * array of nulls fot transferring to constructor
     * @return new object of given class
     * @throws SystemException if some errors occur during object creating such as
     *         absence of constructor with given parameters set
     * @see #loadClass(String)
     */
    public static final Object instantiate(String cName,
                                           Class[] paramClasses,
                                           Object[] params ) throws LogicException, SystemException {
        // Load class
        Class clazz = loadClass(cName);

        try {
            if (paramClasses != null) {
                // Find appropriate constructor
                Constructor clazzConstructor = clazz.getConstructor(paramClasses);

                if ( params == null ) {
                    // assume all parameters as null
                    return clazzConstructor.newInstance(
                            new Object[paramClasses.length]
                    );
                } else {
                    if ( params.length != paramClasses.length ) {
                        // incorrect parameters
                        throw new LogicException("lengths of param and paramClassses arrays must be the same");
                    } else {
                        // use specified parameters
                        return clazzConstructor.newInstance( params );
                    }
                }
            } else {
                // Try to instaciate using parameters only
                return instantiate(cName, params);
            }
        } catch (NoSuchMethodException e) {
            throw new SystemException(e, "Can't instaciate class");
        } catch (SecurityException e) {
            throw new SystemException(e, "Can't instaciate class");
        } catch (InstantiationException e) {
            throw new SystemException(e, "Can't instaciate class");
        } catch (IllegalAccessException e) {
            throw new SystemException(e, "Can't instaciate class");
        } catch (IllegalArgumentException e) {
            throw new SystemException(e, "Can't instaciate class");
        } catch (InvocationTargetException e) {
            throw new SystemException(
                    e.getTargetException(),
                    "Can't instaciate class"
            );
        }
    }


    /**
     * Instanciates class using appropriate constructor and checks if it is of correct class
     *
     * @param cName class name to instantiate
     * @param params class constructor parameters
     * @param clazz class of class to instantiate
     * @return new object of given class
     * @throws SystemException if some errors occur during object creating such as
     *         absence of constructor with given parameters set
     * @throws LogicException if new object is not of clazz class
     */
    public static final Object instantiate(String cName, Object[] params,
                                           Class clazz) throws LogicException, SystemException {
        Object obj = instantiate(cName, params);
        if (!clazz.isInstance(obj)) {
            throw new LogicException("Class " + cName + " is of incorrect class");
        }
        return obj;
    }

    /**
     * Instanciates class using appropriate constructor. Loads class using
     * {@link #loadClass(String)} method.
     *
     * @param cName class name to instantiate
     * @param params class constructor parameter
     * @return new object of given class
     * @throws SystemException if some errors occur during object creating such as
     *         absence of constructor with given parameters set
     * @see #loadClass(String)
     */
    public static final Object instantiate(String cName, Object[] params) throws
            SystemException {
        // Load class
        Class clazz = loadClass(cName);

        try {
            if (params != null) {
                // Construct parameters's classes
                Class[] paramClasses = new Class[params.length];
                for (int i = 0; i < params.length; i++) {
                    paramClasses[i] = params[i].getClass();
                }
                // Find appropriate constructor
                Constructor clazzConstructor = clazz.getConstructor(paramClasses);
                // Try to instanciate instance
                Object obj = clazzConstructor.newInstance(params);
                return obj;
            } else {
                // Instaciate using default constructor
                Object obj = clazz.newInstance();
                return obj;
            }
        } catch (NoSuchMethodException e) {
            throw new SystemException(e, "Can't instaciate class");
        } catch (SecurityException e) {
            throw new SystemException(e, "Can't instaciate class");
        } catch (InstantiationException e) {
            throw new SystemException(e, "Can't instaciate class");
        } catch (IllegalAccessException e) {
            throw new SystemException(e, "Can't instaciate class");
        } catch (IllegalArgumentException e) {
            throw new SystemException(e, "Can't instaciate class");
        } catch (InvocationTargetException e) {
            throw new SystemException(
                    e.getTargetException(),
                    "Can't instaciate class"
            );
        }
    }

    /**
     * Invoke method with given name against given object using given parameters.
     * Returns result of execution according with {@link Method#invoke}.
     *
     * @param o object to invoke method on. Can't be null.
     * @param methodName method name. Can't be empty string.
     * @param parameters method parameters. Used to determine parameters classes.
     * Pass as null or empty array to indicate method without parameters.
     * @return invoke method result.
     * @throws SystemException as wrapper for all exceptions that may be thrown by
     * {@link Method#invoke} or {@link Class#getMethod(String, Class[])}.
     * @see Method#invoke
     * @see Class#getMethod(String, Class[])
     */
    public static final Object invoke(Object o, String methodName, Object[] parameters) throws SystemException {
        Class parameterTypes[];

        if (parameters == null || parameters.length == 0) {
            parameterTypes = new Class[0];
        } else {
            parameterTypes = new Class[parameters.length];
            for (int i = 0; i < parameters.length; i++) {
                parameterTypes[i] = parameters[i].getClass();
            }
        }

        return invoke(o, methodName, parameterTypes, parameters);
    }

    /**
     * Invoke method with given name against given object using given parameters.
     * Returns result of execution according with {@link Method#invoke}.
     *
     * @param o object to invoke method on. Can't be null.
     * @param methodName method name. Can't be empty string.
     * @param parameterTypes parameter types
     * @param parameters method parameters. Used to determine parameters classes.
     * Pass as null or empty array to indicate method without parameters.
     * @return invoke method result.
     * @throws SystemException as wrapper for all exceptions that may be thrown by
     * {@link Method#invoke} or {@link Class#getMethod(String, Class[])}.
     * @see Method#invoke
     * @see Class#getMethod(String, Class[])
     */
    public static final Object invoke(Object o, String methodName, Class[] parameterTypes, Object[] parameters) throws SystemException {
        if (o == null || StringHelper.isEmpty(methodName)) {
            throw new IllegalArgumentException("Object to invoke method on can't be null, method name can't be empty.");
        }

        if ((parameterTypes != null && parameters != null && parameterTypes.length != parameters.length)) {
            throw new IllegalArgumentException("Different length of arrays with parameter types and parameters itself.");
        }

        Class cl = o.getClass();

        try {
            Method method = cl.getMethod(methodName, parameterTypes);

            if (parameters == null) {
                parameters = new Object[0];
            }

            return method.invoke(o, parameters);

        } catch (NoSuchMethodException e) {
            throw new SystemException(e, "Can't invoke method.");
        } catch (SecurityException e) {
            throw new SystemException(e, "Can't invoke method.");
        } catch (IllegalAccessException e) {
            throw new SystemException(e, "Can't invoke method.");
        } catch (IllegalArgumentException e) {
            throw new SystemException(e, "Can't invoke method.");
        } catch (InvocationTargetException e) {
            throw new SystemException(
                    e.getTargetException(),
                    "Exception occured inside the method."
            );
        }
    }

    /**
     * Gets resource with given name with a help of different 
     * classloaders in the following sequence:
     * <li> 1. Thread Context Classloader
     * <li> 2. ClassHelper Classloder
     * <li> 3. System Classloader 
     *
     * @param resource resource name
     * @return resource URL
     */
    public static final URL getResource(String resource) {

        ClassLoader classLoader = null;
        URL url = null;

        try {
            classLoader = Thread.currentThread().getContextClassLoader();

            if (classLoader != null) {
                url = classLoader.getResource(resource);
                if (url != null)
                    return url;
            }

            // We could not find resource. Ler us now try with the
            // classloader that loaded this class.
            classLoader = ClassHelper.class.getClassLoader();
            if (classLoader != null) {
                url = classLoader.getResource(resource);
                if (url != null) {
                    return url;
                }
            }
        } catch (Throwable t) {
            /* do nothing */
        }

        // Last ditch attempt: get the resource from the class path. It
        // may be the case that clazz was loaded by the Extentsion class
        // loader which the parent of the system class loader. Hence the
        // code below.
        return ClassLoader.getSystemResource(resource);
    }

    /**
     * Tries to get specified method from class of the specified object.
     * @param obj Object to search method in
     * @param mName Name of the method to search
     * @param paramTypes Types of parameters for method
     * @return found method
     * @throws SystemException if there is no such method or there are no access rights
     */
    public static final Method getMethod(Object obj, String mName, Class[] paramTypes)
            throws SystemException {
        if (obj == null) {
            throw new IllegalArgumentException("Object to find method in, can't be null");
        }
        if ( StringHelper.isEmpty(mName) == true ) {
            throw new IllegalArgumentException("Method name can't be null or empty");
        }
        try {
            return obj.getClass().getMethod(mName, paramTypes);
        } catch (Exception e) {
            throw new SystemException(e, "Can't get method '" + mName +
                    "' from object (" + obj.getClass().getName() + ")");
        }
    }


    /**
     * Constructs setter name from the name of the property
     * @param propertyName name of property
     * @return
     */
    public static final String createSetterName(String propertyName) {
        if ( StringHelper.isEmpty(propertyName) == true ) {
            throw new IllegalArgumentException("Can't construct setter name");
        }
        char[] tmp = propertyName.toCharArray();
        tmp[0] = Character.toUpperCase( tmp[0] );
        return "set" + new String(tmp);
    }

    /**
     * Constructs getter name from the name of the property
     * @param propertyName name of the property
     * @return
     */
    public static final String createGetterName(String propertyName) {
        if ( StringHelper.isEmpty(propertyName) == true ) {
            throw new IllegalArgumentException("Can't construct setter name");
        }
        char[] tmp = propertyName.toCharArray();
        tmp[0] = Character.toUpperCase( tmp[0] );
        return "get" + new String(tmp);
    }


    /**
     * Template for missing setter method
     */
    private static final String EMSG_TMPL_SETTER_MISSING =
            "Setter for property can't be found (Class:{0}; prop:{1})";

    /**
     * Tries to get setter method for specified property
     * from the specified object and returns type of
     * the setter's argument
     * @param obj instance to search setter in
     * @param propertyName name of the property
     * @return type of the setter's parameter
     * @throws NoSuchMethodException if setter is not found
     */
    public static final Class getSetterParameterType(Object obj, String propertyName)
            throws NoSuchMethodException
    {
        if ( obj == null ) {
            throw new IllegalArgumentException("Specified instance is null");
        }

        String sName = createSetterName(propertyName);

        Method[] methods = obj.getClass().getMethods();
        for (int i = 0; i < methods.length; i++) {
            Method method = methods[i];
            if (
                    (method.getName().equals(sName)) &&
                    (method.getParameterTypes().length == 1)
            ) {
                return method.getParameterTypes()[0];
            }
        }
        throw new NoSuchMethodException(
                MessageFormat.format(
                        EMSG_TMPL_SETTER_MISSING,
                        new Object[] {obj.getClass(), propertyName}
                )
        );
    }

    /**
     * Searches for the field in the class of the specified object and
     * returns type of it.
     * NOTE: Doesn't checks root classes fro the field
     * @param obj instance to search field in
     * @param fieldName name of the field
     * @return Types of the specified property in the specified object
     * @throws NoSuchFieldException if the field is not found
     */
    public static final Class getFieldType(Object obj, String fieldName)
            throws NoSuchFieldException
    {
        if ( obj == null ) {
            throw new IllegalArgumentException("Specified instance is null");
        }
        if ( StringHelper.isEmpty(fieldName) == true ) {
            throw new IllegalArgumentException("Invalid field's name");
        }
        java.lang.reflect.Field field = obj.getClass().getDeclaredField( fieldName );
        return field.getType();
    }

    /**
     * Extracts name of the class only, without package info
     * @param clazz
     * @return name of the class or "null" string if specified
     * clazz is null
     */
    public static final String getClassName(Class clazz)
    {
        if ( clazz == null ) {
            return "null";
        }
        // get full name
        String fullName = clazz.getName();
        // find end of package info
        int lastDotIndex = fullName.lastIndexOf('.');

        if ( lastDotIndex == -1 ) {
            return fullName;
        } else {
            return fullName.substring(lastDotIndex + 1);
        }
    }
    /**
     * Extracts name of the class only, without package info
     * @param obj object from which to get class info
     * @return name of the class or "null" string if specified
     * object is null
     */
    public static final String getClassName(Object obj) {
        if ( obj == null ) {
            return "null";
        } else {
            return getClassName( obj.getClass() );
        }
    }


}
