/*
 * Developed by eVelopers Corporation
 *
 * Copyright (c) 1999-2003 eVelopers Corporation. All rights reserved.
 * This software is the confidential and proprietary information of
 * eVelopers Corporation. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms of
 * the license agreement you entered into with eVelopers.
 *
 * $Date: 04-Feb-04 12:34:17$
 *
 */
package com.evelopers.common.util.property;

import java.io.InputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Properties;
import java.net.URL;

import com.evelopers.common.exception.SystemException;

public class PropertiesConstructor {
	
	/**
	 * Common interface, which will be used in implementation of specific loaders.
	 * Implement it if you want to provide bridg to your loader.
	 */
	public interface IPropertyLoader {
		
		/**
		 * Used to get the name of this loader.
		 * @return name of the this loader
		 */
		public String getLoaderName();
	
		/**
		 * Used to get the value of the specified property from this loader.
		 * @param key name of the requested property
		 * @return value of the requested property or null, if such property was not found in this loader
		 */
		public String getProperty(String key);
	}
	
	/**
	 * Function to construct ready-to-use properties from parametric ones with internal and external links.
	 * 
	 * @param input non-processed properties
	 * @param loaders external loaders to resolve link to external properties
	 * @return processed properties with all replacements
	 * @throws SystemException if any error while properties construction occurres
	 */
	public static Properties parse(Properties input, IPropertyLoader[] loaders)
		throws SystemException {

		Properties output = new Properties();

		Enumeration keys = input.keys();

		while (keys.hasMoreElements()) {
			String key = (String) keys.nextElement();

			// prepare array to store properties' keys, to control self-linked references in the processed one
			ArrayList storage = new ArrayList();
			// key of the current property is also added in this array
			storage.add(key);

			// begin processing of current property
			output = process(key, storage, input, output, loaders);
		}

		return output;
	}

	/**
	 * Function to construct ready-to-use properties from parametric ones with internal links.
	 * @param input non-processed properties
	 * @return processed properties with all replacements
	 * @throws SystemException if any error while properties construction occurres
	 */
	public static Properties parse(Properties input)
		throws SystemException {
		return parse(input, null);
	}

	/**
	 * Function to construct ready-to-use properties from parametric ones with internal links.
	 * Properties for the processing come in form of input stream.
	 * @param stream stream with non-processed properties
	 * @return processed properties with all replacements
	 * @throws SystemException if any error while properties construction occurres
	 */
	public static Properties parse(InputStream stream)
		throws SystemException {
		Properties input = new Properties();

		try {
			input.load(stream);
		} catch (IOException e) {
			throw new SystemException("impossible to load properties from input stream");
		}

		return parse(input, null);
	}

	/**
	 * Function to construct ready-to-use properties from parametric ones with internal and external links.
	 * Properties for the processing come in form of input stream.
	 * @param stream stream of non-processed properties
	 * @param loaders external loaders to resolve link to external properties
	 * @return processed properties with all replacements
	 * @throws SystemException if any error while properties construction occurres
	 */
	public static Properties parse(
		InputStream stream,
		IPropertyLoader[] loaders)
		throws SystemException {
		Properties input = new Properties();

		try {
			input.load(stream);
		} catch (IOException e) {
			throw new SystemException("impossible to load properties from input stream");
		}

		return parse(input, loaders);
	}

	/**
	 * Function to construct ready-to-use properties from parametric ones with internal links.
	 * Properties for the processing come in form of URL.
	 * @param url url with non-processed properties
	 * @return processed properties with all replacements
	 * @throws SystemException if any error while properties construction occurres
	 */
	public static Properties parse(URL url) throws SystemException {
		Properties input = new Properties();

		// Return empty Properties if URL is missing
		if (url == null) {
			return input;
		}

		try {
			input.load(url.openStream());
		} catch (IOException e) {
			throw new SystemException("impossible to load properties from input stream");
		}

		return parse(input, null);
	}

	/**
	 * Function to construct ready-to-use properties from parametric ones with internal and external links.
	 * Properties for the processing come in form of URL.
	 * @param url url with non-processed properties
	 * @param loaders external loaders to resolve link to external properties
	 * @return processed properties with all replacements
	 * @throws SystemException if any error while properties construction occurres
	 */
	public static Properties parse(URL url, IPropertyLoader[] loaders)
		throws SystemException {
		Properties input = new Properties();

		try {
			input.load(url.openStream());
		} catch (IOException e) {
			throw new SystemException("impossible to load properties from input stream");
		}

		return parse(input, loaders);
	}

	/**
	 * service recursive method to process properties.
	 * it replaces link to internal and external properties by their values.
	 * it also performs validation of properties' syntax and maintains control about self-linked properties.
	 * @param key name of the current processed property
	 * @param storage list of properties which are linked with current processed propety
	 * @param input list of input non-processed properties
	 * @param output list of processed properties, which are ready for output
	 * @param loaders array of external loaders to resolve external properties
	 * @throws SystemException if any error while proccessing occurres
	 */
	private static Properties process(
		String key,
		ArrayList storage,
		Properties input,
		Properties output,
		IPropertyLoader[] loaders)
		throws SystemException {
		String value = input.getProperty(key);

		// check if value for the property is missed
		if (value == null) {
			throw new SystemException(
				"value for this property : " + key + " was not found");
		}

		// check if the property has been already defined in the outgoing properties storage
		if (output.getProperty(key) != null) {
			throw new SystemException(
				"this property : " + key + " was already defined");
		}

		while (true) {
			// check current property for simplicity : it must not contain "${" and "}" sequencies
			int i = value.indexOf("${");
			int j = value.indexOf("}");
			if ((i == -1) && (j == -1)) {
				break;
			}

			// check, that quantities of opening and closing sequencies are equal
			int countOpened = countSubStrings(value, "${");
			int countOpenedForMessageFormat = countSubStrings(value, "{");
			int countClosed = countSubStrings(value, "}");
			/* check not only for ${xxx} substrings, but for {xxx} strings also, because
			 * they are being used by MessageFormet.format() method lately */
			if (countOpened != countClosed && countOpenedForMessageFormat != countClosed) {
				throw new SystemException(
					"quantity of opening and closing braces does not match in this property : "
						+ key);
			}
			
			/* break if only {xxx} present in property */
			if (countOpened == 0) {
				break;
			}

			/*@TODO code below is incorrect - it can't process situation, when 
			 * both ${} and {} constructions exist in property */

			// used to store value of processed property (with values of included properties)
			StringBuffer temp = new StringBuffer();

			int begin = 0;
			int end = 0;

			// cycle is performed, while we are going through the value of current property, using "}" as token's delimiter
			while (true) {
				// get position of current closing sequence, and if all were processed then stop the cycle
				end = value.indexOf("}", begin);
				if (end == -1) {
					break;
				}

				// retreive token for processing and increase index for the next search
				String token = value.substring(begin, end);
				begin = end + 1;

				/*
				* if current token doesn't contain opening sequence "${", then simply append it to temporary string
				* and begin processing of next token
				*/
				if (countSubStrings(token, "${") == 0) {
					temp.append(token + "}");
					continue;
				}

				/*
				* retrieve position of last opening sequence in token
				* and get name of most-deeply included property inside current token
				*/
				int opening = token.lastIndexOf("${");
				String variableName = token.substring(opening + 2);
				String variableValue = null;

				/*
				* if name of included variable contains ":" symbol, so it's external property and must be processed separately
				* else process it in common way as standard included property
				*/
				if (variableName.indexOf(":") != -1) {
					variableValue = getExternalProperty(variableName, loaders);
				} else {
					//check if this property has self-linked references
					if (storage.contains(variableName)) {
						throw new SystemException(
							"self-linked reference in this property : " + key);
					}

					// add name of this included property to the array with properties, included in the current property
					storage.add(variableName);

					/*
					* if value for processed included property was not found in the list of processed properties
					* when call this function recursively to get it
					*/
					if (output.getProperty(variableName) == null) {
						process(variableName, storage, input, output, loaders);
					}

					// get value of processed included property, if there were some errors with it, then exception had been throwned beforehand
					variableValue = output.getProperty(variableName);
				}

				// apend to the temporary string the part of current token, which is placed before opening sequence
				temp.append(token.substring(0, opening));

				// apend to the temporary string the value of the processed included variable
				temp.append(variableValue);
			}

			/*
			* if there is some content after last closing sequence in the value of the current property,
			* then add it to the end of temporary string
			*/
			if (begin < value.length()) {
				temp.append(value.substring(begin));
			}

			value = temp.toString();
		}

		// place current property in the storage for processed properties and remove it from storage for incoming ones
		output.put(key, value);
		input.remove(key);

		return output;
	}

	/**
	 * service method to count quantity of sequncies in strings
	 * @return quantity of inclusions of sequence in the string
	 * @param text incoming string for processing
	 * @param substring sequence for count
	 */
	private static int countSubStrings(String text, String substring) {
		int k = 0;
		int count = 0;

		while (text.indexOf(substring, k) != -1) {
			k = text.indexOf(substring, k) + substring.length();
			count++;
		}

		return count;
	}

	/**
	 * service method to process properies from external loaders
	 * @return value of property from external loader
	 * @param key non-processed name of external property
	 * @param loaders array of external loaders to find requested properties
	 * @throws SystemException if any error occures while retrieving an external property
	 */
	private static String getExternalProperty(
		String key,
		IPropertyLoader[] loaders)
		throws SystemException {
		// check if external loaders don't exist
		if (loaders == null) {
			throw new SystemException(
				"there is reference to external property with name : "
					+ key
					+ "  and no external loaders received");
		}

		// check that the syntax of external property is correct
		if ((key.startsWith(":") || key.endsWith(":"))
			|| (countSubStrings(key, ":") != 1)) {
			throw new SystemException("wrong syntax of external property, must be : <loader_name>:<property_name>");
		}

		// parse non-processed name of property to retrieve name of the loader and name loader's internal property
		String loaderName = key.substring(0, key.indexOf(":"));
		String propertyName = key.substring(key.indexOf(":") + 1);

		IPropertyLoader loader = null;

		// cycle to find loader with required name in the incoming array of loaders
		for (int i = 0; i < loaders.length; i++) {
			if (loaders[i].getLoaderName().equals(loaderName)) {
				loader = loaders[i];
				break;
			}
		}

		// check if required loader was found successfully
		if (loader == null) {
			throw new SystemException(
				"loader with such name : " + loaderName + " was not found");
		}

		String value = loader.getProperty(propertyName);

		// check if this value of required property was retrieved successfully
		if (value == null) {
			throw new SystemException(
				"property with name : "
					+ propertyName
					+ " was not found in loader : "
					+ loaderName);
		}

		return value;
	}
}
