/*
 * Developed by eVelopers Corporation
 *
 * Copyright (c) 1999-2003 eVelopers Corporation. All rights reserved.
 * This software is the confidential and proprietary information of
 * eVelopers Corporation. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms of
 * the license agreement you entered into with eVelopers.
 *
 * $Date: 18-Mar-05 11:10:14$
 *
 */
package com.evelopers.common.util.property;

import java.util.*;

/**
 * Helper class to be used for manipulating with Propeties objects.
 * Main functions are:
 * <li>byPrefix()  Finds prefixes with given prefix</li>
 * <li>group()     Groups properties with distinct prefix</li>
 * <li>prefixes()  Finds distinct prefixes</li>
 *
 * @author magus
 * @version $Revision: 2$
 * @see java.util.Properties
 */
public class PropertiesHelper {

   /**
     * Default delimeter character
     */
    public static final char DEF_DELIMETER_CHR = '.';

    /**
     * Finds properties with given prefix using <code>DEF_DELIMETER</code> char as delimeter.
     * <b>NOTE: Cuts prefixes from founded properties</b>
     * @param props Properties to be search
     * @param prefix Prefix to select properties
     * @return found set of properties
     */
    public static final Properties byPrefix(Properties props , String prefix) {
        return byPrefix(props, prefix, true, DEF_DELIMETER_CHR);
    }

    /**
     * Finds properties with given prefix using DEF_DELIMETER char as delimeter.
     * @param props Properties to be search
     * @param prefix Prefix to select properties
     * @param cut Is it necesary to cut off prefixes from founded properties
     * @return found set of properties
     */
    public static final Properties byPrefix(Properties props , String prefix , boolean cut ) {
        return byPrefix(props, prefix, cut, DEF_DELIMETER_CHR);
    }

    /**
     * Finds properties with given prefix.
     * @param props Properties to be search
     * @param prefix Prefix to select properties
     * @param cut Is it necesary to cut off prefixes from founded properties
     * @param delimeter Char to be used as delimeter
     * @return found set of properties
     */
    public static final Properties byPrefix(Properties props , String prefix , boolean cut , char delimeter) {
        // Is properties empty?
        if ( (prefix == null) || (prefix.length() == 0) ) {
            return props;
        }
        // Prepare result properties
        Properties result = new Properties();
        // Construct "full" prefix
        prefix = fullPrefix(prefix , delimeter);
        int pLength = prefix.length();

        // Go through properties
        Iterator iter = props.keySet().iterator();
        while (iter.hasNext()) {
            String name = (String) iter.next();
            if ( name.startsWith( prefix ) == true ) {
                // This property starts from specified prefix; save it
                if ( cut == true ) {
                    result.setProperty( name.substring(pLength) , props.getProperty(name));
                } else {
                    result.setProperty( name , props.getProperty(name));
                }
            }
        }
        return result;
    }

    /**
     * Extends names of all properties in <code>props</code> with <code>prefix</code>
     * using <code>delimiter</code>. 
     * 
     * Note: it is in some way reverse to 
     * <code>PropertiesHelper.byPrefix(.., cut = true, ..)</code>.
     * 
     * @param props
     * @param prefix
     * @param delimeter
     * @return
     */
    public static final Properties extendPrefix(Properties props, String prefix, char delimeter) {
        if ( (prefix == null) || (prefix.length() == 0) ) {
            return props;
        }
        
        // Prepare result properties
        Properties result = new Properties();

        // Construct "full" prefix
      	prefix = fullPrefix(prefix, delimeter);

        // Go through properties
        Iterator iter = props.keySet().iterator();
        while (iter.hasNext()) {
            String name = (String) iter.next();
            result.setProperty(prefix + name, props.getProperty(name));
        }
        return result;
    }

    /**
     * Groups properties using level = 1 to build groups and DEF_DELIMETER_CHR char as delimeter.
     * <b>NOTE: Doesn't cut any prefixes.</b>
     * @param props Properties to be grouped
     * @return map of properties groups. Key for group is prefix, value - set of properties
     *         with this prefix
     */
    public static final Map group(Properties props) {
        return group(props , null , 1 , DEF_DELIMETER_CHR);
    }

    /**
     * Finds properties with specified prefix and groups them with level = 1.
     * Uses DEF_DELIMETER_CHR char as delimeter.
     * @param props Properties to be grouped
     * @param prefix Prefix to select sub-properties which will be grouped
     * @return map of properties groups. Key for group is prefix, value - set of properties
     *         with this prefix
     */
    public static final Map group(Properties props , String prefix) {
        return group(props , prefix , 1 , DEF_DELIMETER_CHR);
    }

    /**
     * Finds properties with specified prefix and groups them.
     * @param props Properties to be grouped
     * @param prefix Prefix to select sub-properties which will be grouped
     * @param level Level of the prefixes (quantity of delimeters) to be used
     *              for grouping of the sub-properties
     * @param delimeter Char to be used as delimeter
     * @return map of properties groups. Key for group is prefix, value - set of properties
     *         with this prefix
     */
    public static final Map group(Properties props , String prefix , int level , char delimeter) {
        Properties cutted = byPrefix(props , prefix , true);
        return group(cutted , level , delimeter);
    }

    /**
     * Groups given properties using prefixes of the given level using DEF_DELIMETER CHR as delimeter.
     * The result is a Map where keys are disctinct prefixes of the specified level
     * and values are "Properties" objects. Mapped properties's kays are constructed using
     * full source keys with cut prefixes.
     * @param props Properties to be grouped
     * @param level Level of the prefixes (quantity of delimeters) to be used for grouping
     * @return map of properties groups. Key for group is prefix, value - set of properties
     *         with this prefix
     */
    public static final Map group(Properties props , int level) {
        return group(props , level , DEF_DELIMETER_CHR);
    }

    /**
     * Groups given properties using prefixes of the given level.
     * The result is a Map where keys are disctinct prefixes of the specified level
     * and values are "Properties" objects. Mapped properties's kays are constructed using
     * full source keys with cut prefixes.
     * @param props Properties to be grouped
     * @param level Level of the prefixes (quantity of delimeters) to be used for grouping
     * @param delimeter Char to be used as delimeter
     * @return map of properties groups. Key for group is prefix, value - set of properties
     *         with this prefix
     */
    public static final Map group(Properties props , int level , char delimeter) {
        // Check props
        if ( props == null ) {
            return new Properties();
        }
        Map result = new HashMap();
        // Get prefixes of the given level
        Set prfxs = prefixes(props, level , delimeter);
        // Go through keys
        Iterator iter = prfxs.iterator();
        while (iter.hasNext()) {
            String prefix = (String) iter.next();
            // Find group for prefix
            Properties group = byPrefix(props , prefix , true);
            result.put( prefix , group);
        }
        return result;
    }

    /**
     * Returns distinct prefixes of the given properties with
     * level = 1 (up to first delimeter) and using DEF_DELIMETER_CHR char
     * as delimeter.
     * @param props Properties to be searched for prefixes
     * @return set of found prefixes
     */
    public static final Set prefixes(Properties props) {
        return prefixes(props , 1 , DEF_DELIMETER_CHR);
    }

    /**
     * Returns distinct prefixes of the given properties with
     * level = 1 (up to first delimeter)
     * @param props Properties to be searched for prefixes
     * @param delimeter Char to be used as delimeter
     * @return set of found prefixes down to first delimeter or first level
     */
    public static final Set prefixes(Properties props , char delimeter) {
        return prefixes(props , 1 , delimeter);
    }

    /**
     * Returns distinct prefixes of the given properties with specified level
     * @param props Properties to be searched for prefixes
     * @param level defines level down to which look for prefixes
     * @return set of found prefixes down to given level
     */
    public static final Set prefixes(Properties props , int level) {
        return prefixes(props , level , DEF_DELIMETER_CHR);
    }

    /**
     * Returns distinct prefixes of the given properties with necessary level
     * @param props Properties to be searched for prefixes
     * @param level Level of the prefix (quantity of delimeters)
     * @param delimeter Char to be used as delimeter
     * @return set of found prefixes down to given level using given delimeter
     */
    public static final Set prefixes(Properties props , int level , char delimeter) {
        Set result = new HashSet();
        // Go through keys
        Iterator iter = props.keySet().iterator();
        while (iter.hasNext()) {
            String name = (String) iter.next();
            String prefix = getPrefix(name , level , delimeter);
            // save prefix in set
            result.add( prefix );
        }
        return result;
    }

    /**
     * Returns normal prefix of the given level from the given string.
     * @param str Source string
     * @param level Level of the prefix (quantity of delimeters)
     * @param delimeter Char to be used as delimeter
     * @return prefix
     */
    private static final String getPrefix(String str , int level , char delimeter) {
        int sIndex = -1;
        // Try to find levelled delimeter
        for (int i = 0; i < level; i++) {
            sIndex = str.indexOf(delimeter , sIndex + 1);
            if ( sIndex == -1 ) {
                break;
            }
        }

        if ( sIndex == -1 ) {
            // Level of the string is less then necessary
            return str;
        } else {
            // return substring (prefix)
            return normalPrefix(str.substring(0 , sIndex + 1) , delimeter);
        }
    }


    /**
     * Constructs "normal" variant of the given prefix.
     * "Normal" variant hasn't delimeter at his end.
     * @param prefix Prefix to be examined
     * @param delimeter Char to be used as delimeter
     * @return prefix
     */
    private static final String normalPrefix(String prefix , char delimeter) {
        // Check if prefix is null
        if ( prefix == null ) {
            return prefix;
        }
        // create string representation of delimeter
        String strDelimeter = new String( new char[] {delimeter} );

        if ( prefix.endsWith(strDelimeter) == true ) {
            return prefix.substring(0 , prefix.length() - strDelimeter.length());
        } else {
            return prefix;
        }
    }

    /**
     * Constructs full prefix from the given one.
     * Full prefix has delimeter char at his end.
     * @param prefix Prefix to be examined
     * @param delimeter Char to be used as delimeter
     * @return prefix
     */
    private static final String fullPrefix(String prefix , char delimeter) {
        // Check if prefix is null
        if ( prefix == null ) {
            return prefix;
        }
        // create string representation of delimeter
        String strDelimeter = new String( new char[] {delimeter} );

        if ( prefix.endsWith(strDelimeter) == true ) {
            return prefix;
        } else {
            StringBuffer tmp = new StringBuffer(prefix);
            tmp.append( strDelimeter );
            return tmp.toString();
        }
    }
}
