\ gpib.4th
\
\ kForth Interface words for the Linux GPIB driver by Frank M. Hess,
\ et al. See the linux-gpib project website on sourceforge:
\
\   http://linux-gpib.sourceforge.net/
\
\ Copyright (c) 1999--2010 Krishna Myneni
\ Provided under the terms of the GNU General Public License
\
\ Requires:
\
\      ans-words.4th
\      ioctl.4th
\      struct.4th
\      struct-ext.4th
\
\ Revisions:
\ 
\	3-22-1999  first working version for Claus Schroeter's
\                  linux-gpib driver
\	3-23-1999  completed port of UR/FORTH GPIB driver
\	5-20-1999  added C_IBTMO and ibtmo timeout words; set
\                  default timeout to 3 seconds.
\       8-07-2006  ported to Frank Hess' linux-gpib driver
\       2-22-2007  added 1 ms delays in send_command, send_bytes, and
\                  read_bytes to accomodate slow GPIB devices  km
\       6-22-2010  revised structures for driver version 3.2.14  km

\ Structures equivalent to those in the driver file gpib_ioctl.h

struct
    100 buf: GPBRD_NAME
end-struct gpib_boardtype%


struct
    int64:  GPRW_BUFFER_PTR
    int: GPRW_COUNT
    int: GPRW_END
    int: GPRW_HANDLE
    int: GPRW_PADDING
end-struct gpib_readwrite%


struct
    int: GPOPEN_HANDLE
    int: GPOPEN_PAD
    int: GPOPEN_SAD
    int: GPOPEN_IS_BOARD
end-struct gpib_opendev%


struct
    int: GPCLOSE_HANDLE
end-struct gpib_closedev%


struct
    int: GPSP_PAD
    int: GPSP_SAD
    int: GPSP_STATUS
end-struct gpib_serialpoll%


struct
    int: GPEOS_EOS
    int: GPEOS_FLAGS
end-struct gpib_eos%


struct
    int: GPWAIT_HANDLE
    int: GPWAIT_WAIT_MASK
    int: GPWAIT_CLEAR_MASK
    int: GPWAIT_SET_MASK
    int: GPWAIT_IBSTA
    int: GPWAIT_PAD
    int: GPWAIT_SAD
    int: GPWAIT_TIMEOUT
end-struct gpib_wait%


struct
    int64: GP_INIT_DATA_PTR
    int:   GP_INIT_DATA
    int:   GP_ONLINE
end-struct gpib_online%


struct
    int: GPSPOLL_NUM_BYTES
    int: GPSPOLL_PAD
    int: GPSPOLL_SAD
end-struct gpib_spollbytes%


struct
    int: GPBD_PAD
    int: GPBD_SAD
    int: GPBD_PPCONFIG
    int: GPBD_AUTOPOLLING
    int: GPBD_IS_SYSCON
    int: GPBD_T1_DELAY
    int: GPBD_IST_NO7BIT
end-struct gpib_boardinfo%





\ GPIB Driver IOCTL numbers, following those defined in the driver file gpib_ioctl.h

160  constant  GPIB_CODE

GPIB_CODE   0  gpib_readwrite% %size  _IOWR  constant  C_IBRD
GPIB_CODE   1  gpib_readwrite% %size  _IOWR  constant  C_IBWRT
GPIB_CODE   2  gpib_readwrite% %size  _IOWR  constant  C_IBCMD
GPIB_CODE   3  gpib_opendev% %size    _IOWR  constant  C_IBOPENDEV
GPIB_CODE   4  gpib_closedev% %size   _IOW   constant  C_IBCLOSEDEV
GPIB_CODE   5  gpib_wait% %size       _IOWR  constant  C_IBWAIT
GPIB_CODE   6  1                      _IOWR  constant  C_IBRPP
GPIB_CODE   9  1 CELLS                _IOW   constant  C_IBSIC
GPIB_CODE  10  1 CELLS                _IOW   constant  C_IBSRE
GPIB_CODE  11                         _IO    constant  C_IBGTS
GPIB_CODE  12  1 CELLS                _IOW   constant  C_IBCAC
GPIB_CODE  14  2                      _IOR   constant  C_IBLINES
GPIB_CODE  17  1 CELLS                _IOW   constant  C_IBTMO
GPIB_CODE  18  gpib_serialpoll% %size _IOWR  constant  C_IBRSP
GPIB_CODE  19  gpib_eos% %size        _IOW   constant  C_IBEOS
GPIB_CODE  20  1                      _IOW   constant  C_IBRSV
GPIB_CODE  26  1 CELLS                _IOW   constant  C_IBMUTEX
GPIB_CODE  29  gpib_boardinfo% %size  _IOR   constant  C_IBBOARD_INFO
GPIB_CODE  33  2                      _IOR   constant  C_IBEVENT
GPIB_CODE  34  1 CELLS                _IOW   constant  C_IBRSC
GPIB_CODE  35  1 CELLS                _IOW   constant  C_IB_T1_DELAY
GPIB_CODE  36                         _IO    constant  C_IBLOC
GPIB_CODE  39  gpib_online% %size     _IOW   constant  C_IBONL


\ ---- Debugging 
0 [IF]
BASE @
HEX
    cr .( linux-gpib driver ioctl codes: ) cr
    cr .( IBRD         ) C_IBRD  u.
    cr .( IBWRT        ) C_IBWRT u.
    cr .( IBCMD        ) C_IBCMD u.
    cr .( IBOPENDEV    ) C_IBOPENDEV u.
    cr .( IBCLOSEDEV   ) C_IBCLOSEDEV u.
    cr .( IBWAIT       ) C_IBWAIT u.
    cr .( IBRPP        ) C_IBRPP u.
    cr .( IBSIC        ) C_IBSIC u.
    cr .( IBSRE        ) C_IBSRE u.
    cr .( IBGTS        ) C_IBGTS u.
    cr .( IBCAC        ) C_IBCAC u.
    cr .( IBLINES      ) C_IBLINES u.
    cr .( IBTMO        ) C_IBTMO u.
    cr .( IBRSP        ) C_IBRSP u.
    cr .( IBEOS        ) C_IBEOS u.
    cr .( IBRSV        ) C_IBRSV u.
    cr .( IBMUTEX      ) C_IBMUTEX u.
    cr .( IBBOARD_INFO ) C_IBBOARD_INFO u.
    cr .( IBEVENT      ) C_IBEVENT u.
    cr .( IBRSC        ) C_IBRSC u.
    cr .( IB_T1_DELAY  ) C_IB_T1_DELAY u.
    cr .( IBLOC        ) C_IBLOC u.
    cr .( IBONL        ) C_IBONL u.
    
(
    Under Linux 2.6.x, the following codes result:
    
IBRD         c010a000
IBWRT        c010a001
IBCMD        c010a002
IBOPENDEV    c010a003
IBCLOSEDEV   4004a004
IBWAIT       c020a005
IBRPP        c001a006
IBSIC        4004a009
IBSRE        4004a00a
IBGTS        a00b
IBCAC        4004a00c
IBLINES      8002a00e
IBPAD        4008a00f
IBSAD        4008a010
IBTMO        4004a011
IBRSP        c00ca012
IBEOS        4008a013
IBRSV        4001a014
IBMUTEX      4004a01a
IBBOARD_INFO 801ca01d
IBEVENT
IBRSC    
IB_T1_DELAY
IBLOC        
IBONL        400ca027
)
BASE !
[THEN]


variable gpib_driver
c" /dev/gpib0" gpib_driver ! 
variable gpib_fd
variable gplock
variable gpremote
variable gptimeout       \ timeout in microseconds
variable gpduration      \ duration to send IFC CLEAR in microsec

create gponl  gpib_online%     %allot drop
create gprw   gpib_readwrite%  %allot drop
create gpinfo gpib_boardinfo%  %allot drop

create ibcmd_buf       64 allot
create gpib_in_buf  16384 allot
create gpib_out_buf 16384 allot

: open_gpib ( -- ior | open the gpib device driver )
    gpib_driver a@ 2 open dup gpib_fd ! 0 < ;

: close_gpib ( -- | close the device driver )
    gpib_fd @ close drop ;

: ibboard_info ( -- error | return board info in )
    gpib_fd @ C_IBBOARD_INFO gpinfo ioctl ;

: iblock ( -- error | lock the board)
    true gplock !
    gpib_fd @ C_IBMUTEX gplock ioctl ;

: ibunlock ( -- error | unlock the board)
    false gplock !
    gpib_fd @ C_IBMUTEX gplock ioctl ;

\ : ibsta ( -- status | return status of last gpib function )
\	ibargs OF_IB_IBSTA + @ ;
\
\ : iberr ( -- error | return error code of last gpib function )
\	ibargs OF_IB_IBERR + @ ;
\
\ : ibcnt ( -- count | return count from last gpib function )
\	ibargs OF_IB_IBCNT + @ ;

\ ibonl requires sysadmin privelage in linux-gpib driver
: ibonl ( b -- error | place the gpib online/offline )
    gponl GP_ONLINE !  0 gponl GP_INIT_DATA !
    gpib_fd @ C_IBONL gponl ioctl ;

: ibsic ( duration -- error | send interface clear on gpib board)
    gpduration !
    gpib_fd @ C_IBSIC gpduration ioctl ;

: ibsre ( b -- error | set or clear remote enable line )
    gpremote ! 
    gpib_fd @ C_IBSRE gpremote ioctl ;

: ibtmo ( v -- error | set timeout to v microseconds)
    gptimeout !
    gpib_fd @ C_IBTMO gptimeout ioctl ;

: ibcmd ( c_n ... c_2 c_1 n -- error | send command bytes to gpib )
    dup gprw GPRW_COUNT !
    0 DO ibcmd_buf i + c! LOOP
    ibcmd_buf gprw GPRW_BUFFER_PTR !
    0 gprw GPRW_HANDLE !
    gpib_fd @ C_IBCMD gprw ioctl ;

: ibrd ( buf u -- error | read u bytes into buf )
    gprw  GPRW_COUNT !
    gprw  GPRW_BUFFER_PTR !
    0 gprw GPRW_HANDLE !
    gpib_fd @ C_IBRD gprw ioctl ;

: ibwrt ( buf u -- error | write u bytes from buf )
    gprw GPRW_COUNT !  gprw GPRW_BUFFER_PTR !
       0 gprw GPRW_HANDLE !
    true gprw GPRW_END !
    gpib_fd @ C_IBWRT gprw ioctl ;

\ ------ end of GPIB primitives

: clear_device ( n -- | send SDC to device at primary address n )
    4 swap 32 + 64 3 ibcmd drop ;

: send_command ( ^str n  -- | send a string to device at primary address n )
    32 + 64 2 ibcmd drop      \ set talker and listener
    1 ms
    count ibwrt drop          \ write data
    1 ms
    95 63 2 ibcmd drop ;      \ untalk and unlisten             

\ send_bytes is similar to send_command except that it uses
\ the output buffer, gpib_out_buf, rather than a counted string.                             
: send_bytes ( u n -- | send u bytes to device at primary address n )
    32 + 64 2 ibcmd drop          \ set talker and listener
    1 ms
    gpib_out_buf swap ibwrt drop  \ write data
    1 ms
    95 63 2 ibcmd drop ;          \ untalk and unlisten       


: read_bytes ( u n -- | read u bytes from device at primary address n )
    64 + 32 2 ibcmd drop       \ set listener and talker
    1 ms
    gpib_in_buf swap ibrd drop \ read data
    1 ms
    63 95 2 ibcmd drop ;       \ untalk and unlisten            

: init_gpib ( -- error | initialize the gpib board and interface )
    \ 1 ibonl 
    iblock 
    100  ibsic  or 
    true ibsre  or
    3000000 ibtmo or
;
