\ minispread.4th
\
\ A small spreadsheet for ANS Forth environments
\
\ From: http://wiki.forthfreak.net/index.cgi?MiniSpreadsheet
\
\ Notes by K. Myneni, 27-Sep-2005:
\
\  1. This program was originally written for gforth. The necessary
\     compatibility code is attached at the beginning to run this code
\     under kForth 1.2.10 or later. The original code has been
\     minimally modified and some extra display functionality has
\     been added.
\
\  2. Cells values are color coded: blue for text, green for calculated cells.
\     The column and row labels are in a grey background for further clarity.
\  
\  3. A help screen has been added. The command key sequence is Esc h
\
\ ========= Needed for kForth ===============
include ans-words
include strings
include ansi
include files

variable read-error

: ptr create 1 cells ?allot ! does> a@ ;

\ The definitions of HERE  ","  "C," are NOT equivalent
\   to the corresponding ANS Forth defs. Their use is
\   restricted to the scope of this program only.
\
CREATE dummy-space 8192 32 * CELLS ALLOT
dummy-space ptr HERE
: ,  ( u|a -- ) HERE  ! HERE CELL+ TO HERE ;
: C, ( u -- )   HERE C! HERE    1+ TO HERE ;
: hallot ( n -- ) here + to here ;

: bounds  ( str len -- str+len str )  over + swap ;
: place  ( caddr n addr -)  2dup  c!  char+  swap  chars  move ;
: off ( addr -- ) false swap ! ;

: toupper ( c -- C ) dup [char] a [char] z 1+ within IF 95 and THEN ;

: ekey    ( -- u | return extended key as concatenated byte sequence )
       BEGIN key? UNTIL
       0 BEGIN  key?  WHILE  8 lshift key or  REPEAT ;

\ Special key codes from ekey
hex
1b5b44   constant left       1b5b43   constant right
1b5b41   constant up         1b5b42   constant down
1b5b367e constant pagedown   1b5b357e constant pageup
1b5b337e constant delete
1b constant esc
decimal

: table ( v1 v2 ... vn n <name> -- | create a table of singles ) 
	create dup cells ?allot over 1- cells + swap
	0 ?do dup >r ! r> 1 cells - loop drop ;

[undefined] floats [if] : floats dfloats ; [then]

: cell- [ 1 cells ] literal - ;

: fdepth ( -- u ) depth 2/ ;

[undefined] >float [if]
: >float ( a u -- r true | false)
   2dup
   over c@ [char] - = IF 1 /string THEN
   dup 0= IF 2drop 2drop false exit THEN
   over c@ [char] . = IF 1 /string THEN
   dup 0= IF 2drop 2drop false exit THEN
   over c@ [char] 0 [char] 9 1+ within >r 2drop r>
   IF strpck string>f true ELSE 2drop false THEN ;
[then]

\ fp number formatting words

: fstring ( f n -- a u | convert f to a formatted string with n decimal places )
    1 swap dup >r 0 ?do 10 * loop s>f f* fround f>d dup -rot dabs
    <# r> 0 ?do # loop [char] . hold #s rot sign #> ; 

: fprint ( f n width -- | print an fp number to n decimal places in width)
    >r fstring r> over - dup 0> IF spaces ELSE drop THEN type ;

: d<> d= invert ;

127 constant BACKSPACE

: edit-line ( a n1 n2 -- n3)        \ only permits backspace editing and appending text
    >r over r@ type r>   ( -- a n1 n2 )
    BEGIN key
      dup BACKSPACE = IF drop 1 cur_left bl emit 1 cur_left 1- 0 max
      ELSE dup 10 = IF drop nip nip exit
      ELSE dup emit swap >r >r over r> swap r@ + c! r> 1+ THEN
      THEN
    AGAIN ;

white constant grey  ( WHITE really appears as a grey color ... )
      
\ ===============end of kForth compatibility defs===========================

decimal   \ Initialize constants.
COLS constant winwd    ROWS constant winht
11 constant slotwd   31 constant strsize
99 constant maxrow   25 constant maxcol

winht 2 -              constant visrows
winwd 2 - slotwd / 1+  constant viscols
maxcol viscols  - 2 +  constant maxcorner_x
maxrow winht    - 3 +  constant maxcorner_y


2 value decimalplaces  2variable currentpos
0 value corner_x  0 value corner_y \ The cell shown in upper, left corner.

maxrow 1+ maxcol 1+ * dup
create farrayadr       floats allot
create sarrayadr strsize 1+ * allot
create copied             256 allot
create mypad              256 allot
2variable from

: farray ( x y -- address) maxcol 1+ * + floats farrayadr + ;
: sarray ( x y -- address) maxcol 1+ * + strsize 1+ * sarrayadr + ;

0 value c
0 ptr adr
0 value n

: cfind ( c cadr n -- n)   ( { c adr n }) to n to adr to c  
  -1 adr n bounds ?DO i c@ c = IF drop i adr - leave THEN LOOP ;

: tidy ( n1 n2 -- larger+1 smaller) 2dup max 1+ -rot min ;
: letter? ( c--flag) toupper [char] A [char] Z 1+ within ;
: digit?  ( c--flag) [char] 0 [char] 9 1+ within ;

: outside? ( x y -- flag) 0 maxrow 1+ within 0= swap 0 maxcol 1+ within 0= or ;
: bottom 0 winht 1- 2dup at-xy  winwd 1- spaces  at-xy ;
: error  bottom type ."  Press a key:"  ekey drop ;

0 value a
0 value b
0 value c
0 value d

: sumrange ( x y x y -- fsum) rot tidy ( { a b }) to b to a  tidy ( { c d }) to d to c
  0.0e  a b do  c d  do  i j farray  f@  f+  loop loop  ;

: ref? ( cadr n -- cadr n flag) over c@ letter? ;

\ Convert slot reference (e.g., "j35") to x,y.

0 ptr adr
0 value n
0 value offset
0 value mul

: ref  ( cadr n -- x y)  [char] A 10 ( { adr n offset mul })
    to mul to offset to n to adr
    n 2 < throw
    adr n bounds ?DO  i c@  toupper offset -  [char] 0 to offset  loop
    n 2 - 0 ?do  swap mul * +  mul 10 * to mul  loop
    2dup outside? IF s" Out-of-bounds reference." error 1 throw THEN ;

0 ptr adr
0 value n
0 value p

: doref ( cadr n -- f)  -1 ( { adr n p } ) to p to n to adr
  [CHAR] : adr n cfind to p ( Range?)
  p -1 > IF  adr p ref  adr p + 1+  n p - 1-  ref? 0= throw  ref  sumrange
  ELSE  adr n ref  farray f@  THEN ;

: ops s" */+-^()" ;

' f*  ' f/  ' f+  ' f-  ' f**  5 table optable

: opfind ( c -- n)  ops cfind ;
: op? ( c -- c flag) dup opfind -1 > ;

create opstack 256 cells allot  opstack cell- ptr sp

: push  sp cell+ to sp  sp ! ;
: copy  sp @ ;  
: pop   copy  sp cell- to sp ;
0 push

: run ( c --) fdepth 2 < throw  opfind cells optable + a@ execute ;

0 value op

: doop  ( { op }) to op   
    op [char] ( = IF op push exit THEN
    BEGIN  copy dup [char] ( <> and  WHILE  pop run  REPEAT
    op [char] ) = IF  pop drop  ELSE  op push  THEN ;


0 ptr adr
0 value op
0 value len

: doval ( op adr -- )   0 ( { op adr len }) to len to adr to op
    here adr -  to len
    len IF  adr len ref? IF  doref  ELSE  >float  0= throw  THEN
    len negate hallot  THEN  op doop ;

0 ptr mark

: infix ( cadr n -- F:f)  here ( { mark  }) to mark
    bounds ?do i c@ op?
     (  ) over  s" +-" cfind 0<   here mark -  or  and ( For unary +-.)
    if  mark doval  else  dup BL = if drop else c, then  then
  loop   0 mark doval  pop drop
  \ Error if operator left on opstack.
  copy IF begin pop drop copy 0= until 1 throw THEN ;

defer afunc
: doarray ( xt --) is afunc maxrow 1+ 0 DO maxcol 1+ 0 DO i j afunc LOOP LOOP ;
: fa!        ( f x y    -- f ) 2>r fdup 2r> farray f! ;
: fillfarray ( f          --) ['] fa! doarray fdrop ;
: sa!        ( cadr n x y --) sarray place ;
: emptyslot  ( x y        --) sarray 0 swap ! ;
: emptysarray  ['] emptyslot  doarray ;

: .empty   slotwd 1- 0 ?DO  ." ." LOOP ;
: pos@ currentpos 2@ ;
: pos! currentpos 2! ;
: >indices ( col row -- i j)  1- corner_y + swap 1- corner_x + swap ;
: ind@  pos@ >indices ;
: .val ( f --) slotwd 1-  decimalplaces ( 2 f.rdp) swap fprint ;
: head ( col --) ?dup IF corner_x + 64 + slotwd 2/ ELSE 32 1 THEN dup >r spaces emit r> spaces ;
: at-slot ( col row --)  swap 1- slotwd * 2 + 0 max swap at-xy ;

0 value x
0 value y

: disp ( x y --)  ( { x y } ) to y to x  
    space  x y sarray count ?dup
    IF  over c@ [char] ' =  \ Label.
      IF pad slotwd blank  1 /string pad swap move pad slotwd 1- 
      blue foreground type
      ELSE
        over c@ [char] = = IF green foreground THEN  
        2drop  x y farray f@ .val  THEN
    ELSE  drop  .empty  THEN 
    black foreground ;

0 value col
0 value row

: (show_slot) ( { col row } ) to row to col 
    col row >indices  col IF ( Print cell value.) disp
    ELSE ( Print row number.)  grey background 2 .r text_normal drop  THEN ;

: show_slot ( col row --)  2dup at-slot  ?dup  IF (show_slot) ELSE grey background head text_normal THEN ;

: showrow ( row --)  0 over at-xy  viscols 0 DO  i over show_slot  LOOP drop ;

: at-current pos@ at-slot ;

: show  winht 1- 0 DO  i showrow  LOOP  at-current ;

0 value n
0 value dn
0 value minimum
0 value ceiling
0 value offset
0 value maxoffset

: bounded ( n dn minimum ceiling offset maxoffset -- n offset)
  ( { n dn minimum ceiling offset maxoffset }) to maxoffset to offset to ceiling to minimum to dn to n
  n dn +  minimum ceiling within
  IF  n dn +  offset  ELSE  n  offset dn +  0 max  maxoffset min  THEN ;

0 value x
0 value y
0 value dx
0 value dy

: move_rel ( dx dy --) \ Move to another cell.
    pos@ ( { dx dy x y } )  to y to x to dy to dx 
    corner_x corner_y  \ Save current view on stack.
    x dx 1 viscols  corner_x maxcorner_x bounded to corner_x
    y dy 1 winht 1- corner_y maxcorner_y bounded to corner_y
    pos!  corner_x corner_y d<> IF ( View has shifted.)  show  THEN
    at-current  ;

: set-dec-places  ( -- ) bottom ." Decimal places? " pad 1 accept pad swap
    -trailing dup
    IF  0 s>d 2swap  >number   nip
      IF s" Bad integer." error 2drop  ELSE  d>s to decimalplaces  THEN
    ELSE 2drop THEN ;

0 ptr adr
0 value n
0 value ch

: calcstr ( cadr n -- f)   0 ( { adr n ch } ) to ch to n to adr   
    n IF  adr c@   to ch
      [char] '  ch =  IF   0.0e   exit  THEN
      [char] =  ch =  IF  adr n  1 /string  infix  exit   THEN
      adr n >float 0= throw
    ELSE   0.0e   THEN ;

: calcslot ( x y --) 2dup 2>r sarray count calcstr 2r> farray f! ;

0 value x
0 value y
0 ptr sadr

: edit ( c --) bottom  ind@  2dup sarray ( { x y sadr }) to sadr to y to x
    x 65 + emit y 1 .r ." >"
    ?dup IF 1 sadr c! sadr 1+ c! ( 1 char. has been typed.) THEN
    sadr count strsize swap edit-line sadr c!  x y ['] calcslot catch  IF 2drop
    s" Bad cell (use = for formulas, ' for labels)." error  0 recurse
    THEN  pos@ show_slot ;

: calc ( col row --) 2dup 2>r sarray  count  calcstr 2r> farray f! ;

: calc_all bottom ." Calculating..." ['] calc doarray
    show bottom ." ...calculated." ;

: update-current  ind@ calcslot  at-current  pos@ show_slot ;
: tocorner  1 1 pos!  at-current ;

0 value p1
0 value vsign
0 value p2

: page_  ( { p1 vsign p2 }) to p2 to vsign to p1
   pos@ drop dup p1 pos! 0 visrows vsign * move_rel
   p2 pos! ;

: page_down  visrows 1 1 page_ ;  
: page_up  1 -1 visrows page_ ;

: 2>str ( n1 n2--cadr u) s>d <# #s bl hold 2drop s>d #s #> ;

0 value handle

: wr ( cadr n--) handle write-line ( throw) drop ;

0 value x
0 value y

: wr-slot ( { x y } ) to y to x 
    x y sarray count ?dup IF x y 2>str wr wr ELSE drop THEN ;
: wr-array ['] wr-slot  doarray ;
: getfname  ( --cadr n) bottom type pad 64 accept pad swap ;


: file-open ( cadr n mode--ior) open-file ?dup IF  nip s" Cannot open file." error
  ELSE  to handle 0  THEN ;

: file-close ( --ior) handle close-file
  IF s" Error while closing file." error THEN ;

: save s" Save to file: " getfname  
    2dup strpck file-exists 
    IF bottom ." File already exists. Overwrite (y/n)? " key toupper 
      [char] Y <> IF 2drop exit THEN 
    THEN w/o O_TRUNC or create-file swap to handle
    IF exit THEN ['] wr-array catch
    IF s" Error while writing to file." error THEN  file-close ;

0 value x

: fload  1 ( { x } ) to x  
    s" Load file: " getfname r/o file-open IF exit THEN
    emptysarray
    BEGIN  pad 80 handle read-line ( n flag ior)
      IF  drop 0  s" Error while reading file." error  THEN
    WHILE pad swap -trailing  x 1 and IF evaluate ELSE 2swap sa! THEN x 1+ to x
    REPEAT  drop  file-close  calc_all ;

variable ,,held   2variable form-offset

: ,,  ( c --)   dup  \ As characters are tacked on, look for slot reference.
  ,,held @
  if  digit?
    if 1 ,,held +!
    else  pad 1+ ,,held @  ref ( x y)
       form-offset 2@ d+   2dup outside?
       if 2drop s" Adjusted reference would be out of bounds." error 1 throw
       then    swap [char] a + c, s>d <# #s #>
       begin dup while over c@ c, 1 /string repeat  2drop  ,,held off
    then
  else  letter? if  1 ,,held !  then
  then   ,,held @ ?dup if  pad + c! else  ?dup if c, then  then ;

\  Adjust slot-references when pasting so that they have the same relation
\  to the current slot that they did to the source slot.

0 ptr mark

: fix-formula ( cadr n -- cadr n)    here ( { mark }) to mark  ,,held off
  ind@  from 2@  d-  form-offset 2!
  bounds  DO  i c@  ,,  LOOP  0 ,,
  mark  here mark -   mypad place    mark here - hallot   mypad count ;

: copy-slot  ind@ 2dup from 2! sarray count copied place bottom ." Copied." ;

: paste  copied count
  ['] fix-formula catch if 2drop copied count then
  ind@ sarray place  update-current ;

: help ( -- )
    CR ." Use cursor arrows and Page Up/Dn to navigate, " 
    CR ." ' to enter text in a cell,  = to enter a formula, "
    CR ." or simply type the number for the current cell. "
    CR
    CR ." Commands are performed with one or two key sequences:"
    CR text_bold
    CR ."   key(s)      Command "
    CR text_normal
    CR ."   Esc q       quit   "
    CR ."   Esc h       show this screen " 
    CR ."   Esc c       calculate all " 
    CR ."   Esc p       set # decimal places " 
    CR ."   Esc s       save spreadsheet " 
    CR ."   Esc l       load spreadsheet " 
    CR ."     ;         copy cell "
    CR ."     ,         paste and move right "
    CR ."     /         paste and move down "
    CR
    CR ." Formulas are not algebraic. Some simple examples (entered using '='): "
    CR
    CR ."   a0:a5      sum cells a0 through a5 "
    CR ."   a5*b2      multiply cell a5 with b2 "
    CR ."   2+5        ... "
    CR
;

: show-help  page cr cr help cr cr ."   Press a key to continue." key drop ;
 
0 value x
0 value m
0 value mess

: mainloop   0 256 0 ( { x m mess } ) to mess to m to x  
    0 copied !  tocorner  show
    BEGIN  ekey ( Get key.)  x or   0 to mess
      CASE
        esc             of  m to x                     endof
	[char] h  m or  of  show-help page show        endof
        [char] q  m or  of  page exit                  endof
        [char] s  m or  of  0 to x  save               endof
        [char] l  m or  of  0 to x  fload              endof
        [char] p  m or  of  0 to x  set-dec-places     endof
        [char] c  m or  of  0 to x  calc_all 1 to mess endof
        [char] ;        of  copy-slot        1 to mess endof
        10              of  0 edit  0  1 move_rel      endof
        left            of         -1  0 move_rel      endof
        right           of          1  0 move_rel      endof
        up              of          0 -1 move_rel      endof
        down            of          0  1 move_rel      endof
        [char] ,        of  paste   1  0 move_rel      endof
        [char] /        of  paste   0  1 move_rel      endof
        pagedown        of  page_down                  endof
        pageup          of  page_up                    endof
        dup 33 128 within  IF dup edit 0 1 move_rel  THEN
      endcase  mess 0= IF  bottom ind@ sarray count type  THEN  at-current
    AGAIN ;

: main  page 0.0e fillfarray emptysarray 0 to corner_x 0 to corner_y
    mainloop ;

CR text_normal
CR .( Spreadsheet program... Type ) text_bold .( MAIN ) text_normal .( to run. ) CR
help
CR CR


