# Copyright (C) 2011-2012, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import socket
import logging
from gettext import gettext as _

import dbus
import gobject
import gconf

from sugar_client import util, env, service
from sugar_client.rest import JSONRest
from sugar_client.util import enforce


INTERFACE = 'org.sugarlabs.client.Server'
NAME = 'org.sugarlabs.client.Server'
PATH = '/org/sugarlabs/client/Server'


class Service(service.Service):

    def __init__(self):
        service.Service.__init__(self, dbus.SessionBus, INTERFACE, NAME, PATH)

    @env.BaseService.method(INTERFACE,
            in_signature='s', out_signature='b')
    def Register(self, api_url):
        if not self.acquire():
            return False

        def register():
            try:
                self._register(api_url or env.api_url.value)
            except Exception, error:
                self.info(_('Cannot process registration: %s'), error)
                util.exception(_('Cannot process registration'))
                self.finished(service.STATE_FAILED)
            else:
                self.info(_('Registration processed successfully'))
                self.finished(service.STATE_SUCCESS)

        gobject.idle_add(register)
        return True

    @property
    def Registered(self):
        return (self.update_status() or {}).get('registered', False)

    @env.BaseService.method(INTERFACE,
            in_signature='', out_signature='')
    def Cancel(self):
        raise NotImplementedError()

    def update_status(self):
        """Fetch status from a server and update local config correspondingly.

        :returns:
            on successful call, returns the status dictionary and None othewise

        """
        if not env.uid.value:
            return

        try:
            rest = JSONRest(env.api_url.value)
            reply = rest.get('/client/status', {'uid': env.uid.value})

            if reply.get('registered'):
                session = env.session()
                for i in ['jabber-url', 'backup-url']:
                    if i in reply:
                        session[i] = reply[i]
                if 'jabber-url' in session:
                    self._set_jabber(session['jabber-url'])
                session['schoolserver'] = socket.getfqdn('schoolserver')
                env.update_session()

            return reply
        except Exception:
            util.exception(_('Cannot connect to school server %s ' \
                    'to fetch client status'), env.api_url.value)

    def _register(self, api_url):
        self.info(_('Registring on %s'), api_url)

        request = {
                'machine_sn': env.get_machine_sn(),
                'machine_uuid': env.get_machine_uuid(),
                'nickname': env.get_nickname(),
                'pubkey': env.pubkey(),
                }
        reply = JSONRest(api_url).post('/client/register', request)

        logging.debug('Got registration reply: %r', reply)
        enforce(reply.get('success') == 'OK',
                _('Registration on %s was failed: %s'),
                api_url, reply.get('error') or _('Unknown error'))

        session = env.session()
        session['api-url'] = api_url
        session['uid'] = reply['uid']
        session['schoolserver'] = socket.getfqdn('schoolserver')
        env.update_session()

    def _set_jabber(self, url):
        self.info(_('Setting Jabber server to %s'), url)

        # Calling sugar-control-panel will trigger TP reconnection
        if util.call(['sugar-control-panel', '-s', 'jabber', url]) is not None:
            return

        logging.warn(_('sugar-control-panel failed, ' \
                'will set jabber directly in GConf'))

        gconf_client = gconf.client_get_default()
        gconf_client.set_string('/desktop/sugar/collaboration/jabber_server',
                url)
