#!/usr/bin/env python

# sugar-lint: disable

import os
import sys
import time
import xmlrpclib
from ConfigParser import ConfigParser
from os.path import exists, relpath, abspath, isfile, isdir

import gtk
import gobject

from __init__ import tests

from sugar_client import env
import sugar_client.backup as backup


class BackupTest(tests.Test):

    def setUp(self):
        tests.Test.setUp(self)
        backup._RSYNC = '/usr/bin/rsync'
        self.mainloop = gobject.MainLoop()
        self.service = backup.Service()

        def on_finished():
            if self.mainloop.is_running():
                self.mainloop.quit()

        self.service.on_finished = on_finished
        env.uid.value = '00000000000000000000000000000000'

    def tearDown(self):
        self.service.remove_from_connection()
        self.service = None
        tests.Test.tearDown(self)

    def touch_rsync(self, cmd):
        self.touch(('rsync', '#!/bin/sh\n' + cmd))
        os.chmod('rsync', 0755)
        backup._RSYNC = tests.tmpdir + '/rsync'

    def test_Backup_LocalDestination(self):
        self.touch('sugar/datastore/test')
        self.service.Backup('dst')
        self.mainloop.run()
        self.assertEqual(0, self.service.State)
        assert exists('dst/test')

    def test_Backup_ExcludeIndex(self):
        self.touch(
                'sugar/datastore/index/',
                'sugar/datastore/index_updated',
                )
        self.service.Backup('dst')
        self.mainloop.run()
        self.assertEqual(0, self.service.State)
        assert not exists('dst/index')
        assert not exists('dst/index_updated')

    def test_Backup_Remote(self):
        self.httpd(8080,
                ('/client/backup?uid=00000000000000000000000000000000', '{"success": "OK", "accepted": true}'),
                )
        env.backup_url.value = 'localhost:/foo'
        env.api_url.value = 'http://localhost:8080'
        self.touch_rsync('touch done')
        self.service.backup(None, 1)
        self.mainloop.run()
        self.assertEqual(0, self.service.State)
        assert exists('done')

    def test_Backup_NoAcceptAfterError(self):
        self.httpd(8080,
                ('/client/backup?uid=00000000000000000000000000000000', 403),
                )
        env.backup_url.value = 'localhost:/foo'
        env.api_url.value = 'http://localhost:8080'
        self.touch_rsync('touch done')
        self.service.backup(None, 1)
        self.mainloop.run()
        self.assertEqual(1, self.service.State)
        assert not exists('done')

    def test_Backup_NoAcceptAfterTries(self):
        self.httpd(8080,
                ('/client/backup?uid=00000000000000000000000000000000', '{"success": "OK", "accepted": false}'),
                ('/client/backup?uid=00000000000000000000000000000000', '{"success": "OK", "accepted": false}'),
                ('/client/backup?uid=00000000000000000000000000000000', '{"success": "OK", "accepted": false}'),
                ('/client/backup?uid=00000000000000000000000000000000', '{"success": "OK", "accepted": true}'),
                )
        env.backup_url.value = 'localhost:/foo'
        env.api_url.value = 'http://localhost:8080'
        backup._BACKUP_SERVER_TIMEOUT = lambda x: x + 1
        self.touch_rsync('touch done')

        self.service.backup(None, 2)
        self.mainloop.run()
        self.assertEqual(1, self.service.State)
        assert not exists('done')

        self.service.backup(None, 2)
        self.mainloop.run()
        self.assertEqual(0, self.service.State)
        assert exists('done')

    def test_Restore(self):
        self.touch(('src/test', 'test'))
        self.service.Restore('src', 0)
        self.mainloop.run()
        assert exists('sugar/datastore/test')
        self.assertEqual('test', file('sugar/datastore/test').read())

    def test_Restore_Default(self):
        self.touch(('src/test', 'test'))
        env.backup_url.value = 'src'
        self.service.Restore(None, 0)
        self.mainloop.run()
        assert exists('sugar/datastore/test')
        self.assertEqual('test', file('sugar/datastore/test').read())

    def test_Restore_DefaultOnDate(self):
        self.touch(('src/1970-01-01/test', 'test'))
        env.backup_url.value = 'src'
        self.service.Restore(None, 1)
        self.mainloop.run()
        assert exists('sugar/datastore/test')
        self.assertEqual('test', file('sugar/datastore/test').read())

    def test_Cancel(self):
        self.touch_rsync('seleep 10')

        self.service.Backup('dst')
        self.service.Cancel()
        ts = time.time()
        self.mainloop.run()
        assert time.time() - ts < 10

        self.service.Restore('src', 0)
        self.service.Cancel()
        ts = time.time()
        self.mainloop.run()
        assert time.time() - ts < 10

    def test_Restore_Reindex(self):

        class Reindexer(object):

            reindexed = []
            trigger = lambda *args: Reindexer.reindexed.append(True)

        self.override(backup, '_Reindexer', Reindexer)
        self.touch(
                'src/',
                )

        self.service.Restore('src', 0)
        self.mainloop.run()
        assert Reindexer.reindexed


if __name__ == '__main__':
    tests.main()
