# Copyright (C) 2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import time
import socket
import datetime
import logging
from os.path import exists, isdir, join
from gettext import gettext as _

import dbus
import gobject

from sugar_client import util, env, service, network


_STARTUP_DELAY = 60 * 5
_FALLBACK_DELAY = 60 * 60
_NETWORK_DETECT_TIMEOUT = 60 * 30
_BACKUP_TRIES = 5

_NM_STATE_CONNECTED = [3, 70]
_NM_STATE_DISCONNECTED = [4]


def start(server, backup, stats=None):
    """Start serving background operations for regular Sugar session."""

    update_task = None
    backup_task = None

    def restored_cb():
        if server.State == service.STATE_SUCCESS:
            backup_task.start()
        else:
            logging.warning(_('Failed to restore, cannot start auto-backup'))

    def registered_cb():
        status = server.update_status() or {}

        if status.get('registered'):
            if status.get('pending-restore') and env.auto_restore.value and \
                    _journal_is_empty():
                backup.on_finished = restored_cb
                backup.Restore('', 0)
            else:
                backup_task.start()
            return

        backup_task.stop()

        if env.auto_register.value and (not env.uid.value or \
                env.schoolserver.value == socket.getfqdn('schoolserver') or \
                env.auto_register.value > 1):
            if server.State == service.STATE_SUCCESS:
                server.Register('')
            else:
                # Repeat in a delay after previously failed registration
                gobject.timeout_add_seconds(_FALLBACK_DELAY,
                        lambda: server.Register('') and False)

    def connected_cb():
        logging.info(_('Network connected'))
        update_task.start()
        server.on_finished = registered_cb
        registered_cb()

    def disconnected_cb():
        logging.info(_('Network disconnected'))
        server.on_finished = None
        update_task.stop()
        backup_task.stop()

    update_task = Scheduler('unattended-update',
            env.update_timeout.value, _FALLBACK_DELAY,
            _unattended_update_cb)
    backup_task = Scheduler('backup',
            env.backup_timeout.value, 0,
            backup.backup, '', _BACKUP_TRIES)
    _connection_manager(connected_cb, disconnected_cb)

    return update_task, backup_task


class Scheduler(object):
    """Repeat callback calling periodically using file based checkpoints."""

    def __init__(self, name, timeout, fallback_delay, cb, *cb_args):
        """
        :param name:
            arbitrary string to name checkpoint file
        :param timeout:
            cast callback in such delay in seconds
        :param fallback_delay:
            if callback is failed, repeat it in such delay in seconds
        :param cb:
            callback to cast periodically
        :param cb_args:
            optional arguments to cast callback

        """
        self._name = name
        self._timeout = timeout
        self._fallback_delay = fallback_delay
        self._cb = cb
        self._cb_args = cb_args
        self._startup_delay = _STARTUP_DELAY
        self._started = None

        self._checkpoint_path = \
                env.profile_path('sugar-client-%s.checkpoint' % name)
        if not exists(self._checkpoint_path):
            file(self._checkpoint_path, 'w').close()

    @property
    def started(self):
        return self._started is not None

    def start(self):
        """Start periodical callback calling."""
        self.stop()
        if not self._timeout:
            return
        self._started = True
        self._run()

    def stop(self):
        """Stop periodical callback calling."""
        if self.started:
            if self._started is not True:
                gobject.source_remove(self._started)
            self._started = None

    def _run(self):
        checkpoint = int(os.stat(self._checkpoint_path).st_mtime)
        startup_delay = self._startup_delay
        self._startup_delay = 0

        if checkpoint <= time.time():
            if startup_delay:
                logging.debug('Make %s seconds delay before ' \
                        'initial start "%s" task', startup_delay, self._name)
                # Keep a pause before startup scheduled process
                # to not overtake machine resources at once
                self._schedule(startup_delay)
                return False
            try:
                logging.debug('Call "%s" scheduled task', self._name)
                self._cb(*self._cb_args)
                # In case if task was stopped from cb()
                if not self.started:
                    return False
            except Exception:
                util.exception(_('Failed to execute "%s" scheduled task'),
                        self._name)
                if self._fallback_delay:
                    # Try once more later
                    logging.debug('Retry calling "%s" in %s seconds',
                            self._name, self._fallback_delay)
                    self._schedule(self._fallback_delay)
                    return False
            checkpoint = time.time() + self._timeout
            os.utime(self._checkpoint_path, (checkpoint, checkpoint))
            logging.debug('New checkpoint for "%s" will be at %s',
                    self._name, datetime.datetime.fromtimestamp(checkpoint))

        self._schedule(checkpoint - time.time())
        return False

    def _schedule(self, seconds):
        timeout = max(1, int(seconds))
        logging.debug('Trigger "%s" in %s second(s)', self._name, timeout)
        self._started = gobject.timeout_add_seconds(timeout, self._run)


def _unattended_update_cb():
    bus = dbus.SystemBus()
    system = dbus.Interface(
            bus.get_object('org.sugarlabs.client.System',
                '/org/sugarlabs/client/System'),
            'org.sugarlabs.client.System')
    system.Update()


def _connection_manager(connected_cb, disconnected_cb):
    current_state = [False]

    def connected():
        if current_state[0]:
            return
        current_state[0] = True
        connected_cb()

    def disconnected():
        if not current_state[0]:
            return
        current_state[0] = False
        disconnected_cb()

    def NetworkManagerStateChanged_cb(state):
        if state in _NM_STATE_CONNECTED:
            connected()
        elif state in _NM_STATE_DISCONNECTED:
            disconnected()

    def nm_less_manager():
        if network.is_online():
            connected()
        else:
            disconnected()
        return True

    try:
        bus = dbus.SystemBus()
        obj = bus.get_object('org.freedesktop.NetworkManager',
                '/org/freedesktop/NetworkManager')
        props = dbus.Interface(obj, dbus.PROPERTIES_IFACE)
        state = props.Get('org.freedesktop.NetworkManager', 'State')
        NetworkManagerStateChanged_cb(state)
        bus.add_signal_receiver(NetworkManagerStateChanged_cb,
                'StateChanged', 'org.freedesktop.NetworkManager')
    except dbus.DBusException:
        nm_less_manager()
        gobject.timeout_add_seconds(_NETWORK_DETECT_TIMEOUT, nm_less_manager)


def _journal_is_empty():
    root = env.profile_path('datastore')
    if exists(root):
        for i in os.listdir(root):
            if len(i) == 2 and isdir(join(root, i)):
                return False
    return True
