# Copyright (C) 2012, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

# pylint: disable-msg=E1101,E0102

import os
import zipfile
from cStringIO import StringIO
from ConfigParser import ConfigParser
from os.path import exists, join
from gettext import gettext as _

import active_document as ad
import restful_document as rd
enforce = ad.util.enforce

from sugar_network_server import env
from sugar_network_server.licenses import GOOD_LICENSES
from sugar_network_server.resources.resource import Resource


_ASLO_DOWNLOAD_URL = 'http://download.sugarlabs.org/activities'
_ASLO_ACTIVITIES_PATH = '/upload/activities'


class Implementation(Resource):

    @ad.active_property(prefix='C',
            permissions=ad.ACCESS_CREATE | ad.ACCESS_READ)
    def context(self, value):
        return value

    @ad.active_property(prefix='L', typecast=[GOOD_LICENSES],
            permissions=ad.ACCESS_CREATE | ad.ACCESS_READ)
    def license(self, value):
        return value

    @ad.active_property(slot=2, prefix='V',
            permissions=ad.ACCESS_CREATE | ad.ACCESS_READ)
    def version(self, value):
        return value

    @ad.active_property(slot=3, prefix='D',
            permissions=ad.ACCESS_CREATE | ad.ACCESS_READ, typecast=int)
    def date(self, value):
        return value

    @ad.active_property(slot=4, prefix='S',
            permissions=ad.ACCESS_CREATE | ad.ACCESS_READ,
            typecast=env.STABILITIES)
    def stability(self, value):
        return value

    @ad.active_property(ad.StoredProperty, typecast=dict, default={})
    def feed(self, value):
        return value

    @feed.setter
    def feed(self, value):
        value.setdefault('implementations', {})
        value['implementations'].setdefault('*-*', {})
        return value

    @ad.active_property(full_text=True,
            permissions=ad.ACCESS_CREATE | ad.ACCESS_READ)
    def notes(self, value):
        return value

    @ad.active_property(ad.BlobProperty)
    def bundle(self, value):
        return value

    def recv_blob(self, prop, url):
        if not url.startswith(_ASLO_DOWNLOAD_URL):
            return Resource.recv_blob(self, prop, url)

        path = url[len(_ASLO_DOWNLOAD_URL):].strip('/').split('/')
        enforce(len(path) == 2 and path[-1].endswith('.xo'),
                _('Incorrect activities.sugarlabs.org path'))

        path = join(_ASLO_ACTIVITIES_PATH, *path)
        if exists(path):
            try:
                zp = zipfile.ZipFile(path)
                extract = zp.namelist()[0].split(os.sep)[0]

                activity_info_data = StringIO(
                        zp.read(join(extract, 'activity', 'activity.info')))
                activity_info = ConfigParser()
                activity_info.readfp(activity_info_data)

                self.feed['commands'] = {
                        'activity': activity_info.get('Activity', 'exec'),
                        }
            except Exception, error:
                ad.util.exception()
                raise RuntimeError(_('Cannot read bundle: %s') % error)

        self.feed['implementations']['*-*']['url'] = url
        self.post()

    def send_blob(self, prop):
        url = self.feed['implementations']['*-*'].get('url')
        if url:
            raise rd.SeeOther(url)
        return Resource.send_blob(self, prop)
