# Copyright (C) 2012, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import logging
from ConfigParser import ConfigParser
from os.path import join, exists
from gettext import gettext as _

import gio
import gobject

from sugar_stats import util
from sugar_stats.sniffer import Sniffer


_logger = logging.getLogger('sugar_stats')


class Files(Sniffer):

    _monitors = {}

    def start(self):
        self._monitor_file(_sugar_profile_path('friends'), self.__friends_cb)

    def stop(self):
        while self._monitors:
            path, (hid, monitor) = self._monitors.popitem()
            _logger.debug('Stop monitoring %s', path)
            gobject.source_remove(hid)
            monitor.cancel()

    def _monitor_file(self, path, callback):
        if exists(path):
            callback(path)
        else:
            callback(None)
        monitor = gio.File(path).monitor_file()
        hid = monitor.connect('changed', self.__monitor_changed_cb, callback)
        self._monitors[path] = (hid, monitor)

    def __monitor_changed_cb(self, monitor, one_file, other_file, event_type,
            callback=None):
        path = one_file.get_path()
        if event_type in (gio.FILE_MONITOR_EVENT_CREATED,
                gio.FILE_MONITOR_EVENT_CHANGES_DONE_HINT):
            if exists(path):
                callback(path)
            else:
                callback(None)
        elif event_type == gio.FILE_MONITOR_EVENT_DELETED:
            callback(None)

    def __friends_cb(self, path):
        friends = 0
        if path:
            try:
                config = ConfigParser()
                config.read(path)
                friends = len(config.sections())
            except Exception:
                util.exception(_('Cannot read friends file'))
        self.stats.update('shell.friends', friends)


def _sugar_profile_path(*args):
    profile_dir = join(os.environ['HOME'], '.sugar',
            os.environ.get('SUGAR_PROFILE', 'default'))
    if not exists(profile_dir):
        os.makedirs(profile_dir, 0770)
    return join(profile_dir, *args)
