# Copyright (C) 2012, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import logging

from sugar_stats import dbus_monitor
from sugar_stats.sniffer import Sniffer


_ACTIVITY_IFACE = 'org.laptop.Activity'

_logger = logging.getLogger('sugar_stats')


class Presence(Sniffer):

    _bus = dbus_monitor.session()
    _activities = {}
    _activities_by_path = {}
    _unknown_channels = {}

    def start(self):
        self._bus.connect_to_name(self.__activity_name_cb)

        # http://telepathy.freedesktop.org/spec/
        #   Connection.html#Method:RequestChannel
        self._bus.connect(type='method_call',
                interface='org.freedesktop.Telepathy.Connection',
                member='RequestChannel',
                callback=self.__CreateChannel_cb)
        # http://telepathy.freedesktop.org/spec/
        #   Connection_Interface_Requests.html#Method:CreateChannel
        self._bus.connect(type='method_call',
                interface='org.freedesktop.Telepathy.Connection',
                member='CreateChannel',
                callback=self.__CreateChannel_cb)

        # http://telepathy.freedesktop.org/spec/
        #   Channel_Interface_Group.html#Signal:MembersChanged
        self._bus.connect(type='signal',
                interface='org.freedesktop.Telepathy.Channel.Interface.Group',
                member='MembersChanged',
                callback=self.__MembersChanged_cb)

    def stop(self):
        self._bus.disconnect_by_func(self.__activity_name_cb)
        self._bus.disconnect_by_func(self.__CreateChannel_cb)
        self._bus.disconnect_by_func(self.__CreateChannel_cb)
        self._bus.disconnect_by_func(self.__MembersChanged_cb)

        self._activities.clear()
        self._activities_by_path.clear()
        self._unknown_channels.clear()

    def __activity_name_cb(self, name, new, old):
        if not name.startswith(_ACTIVITY_IFACE):
            return
        if new and not old:
            activity = self._activities[new] = _Activity(self.stats, name)
            _logger.debug('Activity %s appeared on dbus', activity)
        elif old and not new:
            activity = self._activities.get(old)
            if activity is not None:
                if activity.path:
                    del self._activities_by_path[activity.path]
                del self._activities[old]
                activity.close()
                _logger.debug('Activity %s disappeared from dbus', activity)

    def __CreateChannel_cb(self, msg):
        activity = self._activities.get(msg.get_sender())
        if activity is None:
            return

        args = msg.get_args_list()
        if args[0] != 'org.freedesktop.Telepathy.Channel.Type.Text':
            return

        _logger.debug('Activity %s requested text channel', activity)

        self._bus.connect(self.__CreateChannel_reply_cb,
                msg.get_serial(), activity,
                type='method_return', sender=msg.get_destination(),
                destination=msg.get_sender())

    def __CreateChannel_reply_cb(self, msg, serial, activity):
        if msg.get_reply_serial() != serial:
            return

        activity.path = msg.get_args_list()[0]
        self._activities_by_path[activity.path] = activity

        _logger.debug('Activity %s got %s text channel',
                activity, activity.path)

        if activity.path in self._unknown_channels:
            added, removed = self._unknown_channels.pop(activity.path)
            activity.update_buddies(added, removed)

        return True

    def __MembersChanged_cb(self, msg):
        args = msg.get_args_list()
        added = len(args[1])
        removed = len(args[2])

        activity = self._activities_by_path.get(msg.get_path())
        if activity is None:
            self._unknown_channels[msg.get_path()] = (added, removed)
        else:
            if activity.path in self._unknown_channels:
                del self._unknown_channels[activity.path]
            activity.update_buddies(added, removed)


class _Activity(object):

    def __init__(self, stats, name):
        self._stats = stats
        self._bundle_id = None

        self.activity_id = name[len(_ACTIVITY_IFACE):]
        self.path = None
        self.buddies = None

    def __repr__(self):
        return self.activity_id

    @property
    def bundle_id(self):
        if self._bundle_id is None:
            x11 = self._stats.sniffer('x11')
            x11_activity = x11.activities.get(self.activity_id)
            if x11_activity is not None:
                self._bundle_id = x11_activity.bundle_id
        return self._bundle_id

    def close(self):
        if self.buddies:
            self._stats.update('activity.buddies', - self.buddies,
                    self.bundle_id)
            self.buddies = 0

    def update_buddies(self, added, removed):
        if added:
            if self.buddies is None:
                # Exclude activity launcher
                added -= 1
                self.buddies = 0
            self._stats.update('activity.buddies', added, self.bundle_id)
            self.buddies += added

        if removed and self.buddies is not None:
            removed = min(self.buddies, removed)
            self._stats.update('activity.buddies', - removed, self.bundle_id)
            self.buddies -= removed
