# Copyright (C) 2012, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from os.path import join
from gettext import gettext as _

import restful_document as rd
enforce = rd.util.enforce

from sugar_stats_server import env
from sugar_stats_server.rrd import Rrd


class User(rd.User):

    _rrd = {}

    @rd.restful_method(method='GET', cmd='stats-info')
    def _stats_info(self):
        enforce(rd.principal.user == self.guid,
                _('Operation is permitted only for owner'))

        status = {}
        rrd = User._get_rrd(self.guid)
        for name, __, last_update in rrd.dbs:
            status[name] = last_update + env.stats_step.value

        return {'enable': env.stats.value,
                'step': env.stats_step.value,
                'rras': env.stats_client_rras.value,
                'status': status,
                }

    @rd.restful_method(method='POST', cmd='stats-upload')
    def _stats_upload(self):
        enforce(rd.principal.user == self.guid,
                _('Operation is permitted only for owner'))
        rrd = User._get_rrd(self.guid)
        content = rd.request.content
        for timestamp, values in content['values']:
            rrd.put(content['name'], values, timestamp)

    @classmethod
    def _get_rrd(cls, guid):
        rrd = cls._rrd.get(guid)
        if rrd is None:
            rrd = cls._rrd[guid] = Rrd(
                    join(env.stats_root.value, guid[:2], guid),
                    env.stats_step.value, env.stats_server_rras.value)
        return rrd
