# Copyright (C) 2011-2012, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import hashlib
import urlparse
from os.path import join, exists, dirname
from gettext import gettext as _

import gconf
import dbus.service

from sugar_client import util
from sugar_client.util import enforce
from sugar_client.confile import Confile


VERSION = '1.0'

debug = util.Option(
        _('debug logging level; multiple argument'),
        default=0, type_cast=int, short_option='-D', action='count')

uid = util.Option(
        _('current user\'s unique identity; needs to be set only from ' \
                'profile configuration file; will be auto-created ' \
                'on initial registration'))

schoolserver = util.Option(
        _('school server\'s FQDN, will be auto-set after registration'))


_XO_SERIAL_PATH = '/ofw/mfg-data/SN'
_XO_UUID_PATH = '/ofw/mfg-data/U#'
_NICKNAME_GCONF = '/desktop/sugar/user/nick'

_session = None


def _api_url_cast(url):
    if not url.startswith('http://') and not url.startswith('https://'):
        url = 'http://' + url
    url = urlparse.urlparse(url)
    # pylint: disable-msg=E1101
    return '%s://%s' % (url.scheme, url.netloc)


api_url = util.Option(
        _('url to connect to school server\'s RESTfull API'),
        default='http://schoolserver:8000', type_cast=_api_url_cast,
        short_option='-a')

auto_register = util.Option(
        _('process auto-registeration on a school server on startup; ' \
                'multiple argument, if mentioned more than once and ' \
                'previously registered server\'s IP is different from ' \
                '"schoolserver"\'s one, process re-registrtion'),
        default=0, type_cast=int, action='count')

jabber_url = util.Option(
        _('jabber server for Sugar collaboraiton'))

backup_url = util.Option(
        _('rsync url to backup/restore Journal'))

backup_timeout = util.Option(
        _('if more than 0, do regular backups with specified delay ' \
                'in seconds'),
        default=0, type_cast=int)

auto_restore = util.Option(
        _('if, after registration, there are backups to restore on the ' \
                'server, process restore automatically'),
        default=False, action='store_true')

update_args = util.Option(
        _('optional command-line arguments to pass to a packager ' \
                'while updating the system'))

update_timeout = util.Option(
        _('if more than 0, do regular unattended updates with specified ' \
                'delay in seconds'),
        default=0, type_cast=int)

stats_url = util.Option(
        _('stats server\'s http(s) url to interact with; if omited, ' \
                'any stats related functionality is disabled'))

stats_ca_certs = util.Option(
        _('CA certificates file to get access to --stats-url via HTTPS'))

stats_timeout = util.Option(
        _('if more than 0 and --stats-url specified, ping stats server ' \
                'with specified delay in seconds to coordinate ' \
                'stats gathering work'),
        default=3600, type_cast=int)

machine_sn = util.Option(
        _('specify machine\'s serial number instead of detecting it'))

machine_uuid = util.Option(
        _('specify machine\'s UUID instead of detecting it'))

nickname = util.Option(
        _('specify user\'s nickname instead of detecting it'))


def profile_path(*args):
    """Path within sugar profile directory.

    Missed directories will be created.

    :param args:
        path parts that will be added to the resulting path
    :returns:
        full path with directory part existed

    """
    if os.geteuid():
        root_dir = join(os.environ['HOME'], '.sugar',
                os.environ.get('SUGAR_PROFILE', 'default'))
    else:
        root_dir = '/var/sugar-client'
    result = join(root_dir, *args)
    if not exists(dirname(result)):
        os.makedirs(dirname(result))
    return result


def has_pubkey():
    return exists(profile_path('owner.key.pub'))


def pubkey():
    pubkey_path = profile_path('owner.key.pub')
    enforce(exists(pubkey_path),
            _('Sugar session was never started, no pubkey'))

    for line in file(pubkey_path).readlines():
        line = line.strip()
        if line.startswith('ssh-'):
            return line
    raise RuntimeError(_('Valid SSH public key was not found in %s') % \
            pubkey_path)


def get_nickname():
    if nickname.value:
        return nickname.value
    gconf_client = gconf.client_get_default()
    return gconf_client.get_string(_NICKNAME_GCONF)


def get_machine_sn():
    if machine_sn.value:
        return machine_sn.value
    if exists(_XO_SERIAL_PATH):
        return _read_XO_value(_XO_SERIAL_PATH)


def get_machine_uuid():
    if machine_uuid.value:
        return machine_uuid.value
    if exists(_XO_UUID_PATH):
        return _read_XO_value(_XO_UUID_PATH)


def pubkey_hash():
    key = pubkey().split()[1]
    return str(hashlib.sha1(key).hexdigest())


def session():
    """Returns `ConfigParser` object with session config."""
    global _session
    if _session is None:
        config_path = profile_path('sugar-client.conf')
        _session = Confile()
        if exists(config_path):
            _session.read(config_path)
    return _session


def update_session():
    """Update persistant session config.

    This function will update particular `env.*` config variables.

    """
    session_config = session()
    if session_config.has_section(Confile.DEFAULT_SECTION):
        for key, value in session_config.items(Confile.DEFAULT_SECTION):
            if key in util.Option.items:
                util.Option.items[key].value = value
    config_file = util.new_file(profile_path('sugar-client.conf'))
    session_config.write(config_file)
    config_file.close()


class LocalService(object):

    def __init__(self, bus_class, name, path):
        pass

    def remove_from_connection(self, connection=None, path=None):
        pass

    @staticmethod
    def signal(*args, **kwargs):

        def decorate(f):
            return f

        return decorate

    @staticmethod
    def method(*args, **kwargs):

        def decorate(f):
            return f

        return decorate

    def get_http_proxy(self, sender):
        return os.environ.get('http_proxy')


class DBusService(dbus.service.Object):

    def __init__(self, bus_class, name, path):
        bus = bus_class()
        if name:
            bus_name = dbus.service.BusName(name, bus=bus)
            dbus.service.Object.__init__(self, bus_name, path)
        else:
            dbus.service.Object.__init__(self, bus, path)

    @staticmethod
    def signal(*args, **kwargs):
        return dbus.service.signal(*args, **kwargs)

    @staticmethod
    def method(*args, **kwargs):
        return dbus.service.method(*args, **kwargs)

    def get_http_proxy(self, sender):
        dbus_info = dbus.Interface(
                self.connection.get_object('org.freedesktop.DBus',
                    '/org/freedesktop/DBus/Bus'),
                'org.freedesktop.DBus')
        pid = dbus_info.GetConnectionUnixProcessID(sender)
        environ = file('/proc/%s/environ' % pid).read()
        http_proxy = [i.split('=', 1)[-1] for i in environ.split('\0') \
                if i.startswith('http_proxy=')]
        return http_proxy[0] if http_proxy else None


def _read_XO_value(path):
    return file(path).read().rstrip('\0\n')


BaseService = LocalService

util.Option.seek('main')
util.Command.seek('main')
