#!/usr/bin/env python

# sugar-lint: disable

import os
import sys
import xmlrpclib
from ConfigParser import ConfigParser

import gconf
import gobject

from __init__ import tests

from sugar_client import env, server


class ServerTest(tests.Test):

    def setUp(self):
        tests.Test.setUp(self)
        self.mainloop = gobject.MainLoop()
        self.service = server.Service()

        def on_finished():
            if self.mainloop.is_running():
                self.mainloop.quit()

        self.service.on_finished = on_finished
        env.uid.value = '00000000000000000000000000000000'

        self._orig_PATH = os.environ['PATH']
        os.environ['PATH'] = '/bin:/sbin'

    def tearDown(self):
        os.environ['PATH'] = self._orig_PATH
        self.service.remove_from_connection()
        self.service = None
        tests.Test.tearDown(self)

    def test_Register_ErrorWOKeys(self):
        self.service.Register('http://localhost:8080')
        self.mainloop.run()
        self.assertEqual(1, self.service.State)

    def test_Register(self):
        self.httpd(8080,
                ('POST:/client/register', '{"success": "OK", "uid": "UID"}'),
                )
        self.touch(('sugar/owner.key.pub', 'ssh-dss pubkey'))
        env.api_url.value = None
        env.uid.value = None

        self.service.Register('http://localhost:8080')
        self.mainloop.run()
        self.assertEqual(0, self.service.State)
        self.assertEqual('http://localhost:8080', env.api_url.value)
        self.assertEqual('UID', env.uid.value)

        config = ConfigParser()
        config.read('sugar/sugar-client.conf')

        self.assertEqual('http://localhost:8080', config.get('main', 'api-url'))
        self.assertEqual('UID', config.get('main', 'uid'))

    def test_Register_CreateUID(self):
        self.httpd(8080,
                ('POST:/client/register', '{"success": "OK", "uid": "UID"}'),
                )
        self.touch(('sugar/owner.key.pub', 'ssh-dss pubkey'))

        env.uid.value = None
        self.service.Register('http://localhost:8080')
        self.mainloop.run()
        self.assertEqual(0, self.service.State)
        self.assertEqual('UID', env.uid.value)
        assert env.uid.value

        config = ConfigParser()
        config.read('sugar/sugar-client.conf')
        self.assertEqual(env.uid.value, config.get('main', 'uid'))

    def test_Register_DoNotCreateUIDOnFailedRegistration(self):
        self.httpd(8080,
                ('POST:/client/register', '{"success": "ERR", "error": "foo"}'),
                ('POST:/client/register', '{"success": "ERR", "error": "foo"}'),
                )
        self.touch(('sugar/owner.key.pub', 'ssh-dss pubkey'))

        env.uid.value = None
        self.service.Register('http://localhost:8080')
        self.mainloop.run()
        self.assertNotEqual(0, self.service.State)
        self.assertEqual(None, env.uid.value)

        env.uid.value = 'fake'
        self.service.Register('http://localhost:8080')
        self.mainloop.run()
        self.assertNotEqual(0, self.service.State)
        self.assertEqual('fake', env.uid.value)

    def test_update_status(self):
        self.touch(('sugar-control-panel', ['#!/bin/sh', 'echo $@ > sugar-control-panel.input']))
        os.chmod('sugar-control-panel', 0755)
        os.environ['PATH'] = ':'.join([tests.tmpdir, os.environ['PATH']])

        self.httpd(8080,
                ('/client/status?uid=UID', '{' \
                        '"registered": true,' \
                        '"jabber-url": "test_jabberserver",' \
                        '"backup-url": "test_backupurl"' \
                        '}'),
                )
        self.touch(('sugar/owner.key.pub', 'ssh-dss pubkey'))

        env.api_url.value = 'http://localhost:8080'
        env.uid.value = 'UID'
        registered = self.service.Registered
        self.assertEqual(True, registered)
        self.assertEqual('test_jabberserver', env.jabber_url.value)
        self.assertEqual('test_backupurl', env.backup_url.value)

        self.assertEqual(
                '-s jabber test_jabberserver\n',
                file('sugar-control-panel.input').read())

        config = ConfigParser()
        config.read('sugar/sugar-client.conf')
        self.assertEqual('test_jabberserver', config.get('main', 'jabber-url'))
        self.assertEqual('test_backupurl', config.get('main', 'backup-url'))

    def test_update_status_FallbackToGConfForJabber(self):

        class GConfClient(object):

            key = None
            value = None

            def set_string(self, key, value):
                GConfClient.key = key
                GConfClient.value = value

        self.override(gconf, 'client_get_default', GConfClient)

        self.httpd(8080,
                ('/client/status?uid=UID', '{' \
                        '"registered": true,' \
                        '"jabber-url": "test_jabberserver",' \
                        '"backup-url": "test_backupurl"' \
                        '}'),
                )
        self.touch(('sugar/owner.key.pub', 'ssh-dss pubkey'))

        env.api_url.value = 'http://localhost:8080'
        env.uid.value = 'UID'
        registered = self.service.Registered
        self.assertEqual(True, registered)

        self.assertEqual('/desktop/sugar/collaboration/jabber_server', GConfClient.key)
        self.assertEqual('test_jabberserver', GConfClient.value)

    def test_Registered(self):
        self.httpd(8080,
                ('/client/status?uid=UID', 403),
                ('/client/status?uid=UID', '{"success": "OK", "registered": false}'),
                ('/client/status?uid=UID', '{"success": "OK", "registered": true}'),
                )

        env.uid.value = None
        env.api_url.value = 'localhost:8080'
        self.assertEqual(False, self.service.Registered)

        env.uid.value = 'UID'
        self.assertEqual(False, self.service.Registered)
        self.assertEqual(False, self.service.Registered)
        self.assertEqual(True, self.service.Registered)


if __name__ == '__main__':
    tests.main()
