#!/usr/bin/env python

import os
import time
import signal
import getpass

import MySQLdb as mdb

import restful_document
import sugar_network as client
import sugar_network_server as server


EXCLUDE_BUNDLE_IDS = ['net.gcompris']

CATEGIORIES_TO_TAGS = {
        'Search & Discovery': 'discovery',
        'Documents': 'documents',
        'Chat, mail and talk': 'conversation',
        'Programming': 'programming',
        'Maps & Geography': 'geography',
        'Media players': 'media',
        'Teacher tools': 'teacher',
        'Games': 'games',
        'Media creation': 'media',
        'Maths & Science': 'science',
        'News': 'news',
        }

MISNAMED_LICENSES = {
        ('artistic', '2.0'): 'Artistic 2.0',
        ('cc-by-sa',): 'CC-BY-SA',
        ('creative', 'share', 'alike'): 'CC-BY-SA',
        }

connection = None
sugar_guid = None


def main():
    global connection, sugar_guid

    connection = mdb.connect('localhost',
            'root', getpass.getpass(), 'activities')

    query = client.Context.find(title='Sugar')
    if query.total:
        sugar_guid = query[0]['guid']
    else:
        context = client.Context()
        context['type'] = 'application'
        context['title'] = 'Sugar'
        context['summary'] = 'Sugar'
        context['description'] = 'Sugar'
        context.post()
        sugar_guid = context['guid']

        for version in ['0.82', '0.84', '0.86', '0.88', '0.90', '0.92',
                '0.94', '0.96']:
            impl = client.Implementation()
            impl['context'] = sugar_guid
            impl['license'] = ['GPLv3+']
            impl['version'] = version
            impl['date'] = 0
            impl['stability'] = 'stable'
            impl['notes'] = ''
            impl.post()

    import_versions()


def import_versions():
    rows = sqlexec("""
        SELECT
            addons.id,
            addons.guid,
            licenses.name,
            (select max(localized_string) from translations where
                id=licenses.text),
            versions.created,
            versions.version,
            (select max(localized_string) from translations where
                id=versions.releasenotes),
            files.filename,
            (select version from appversions where
                id=applications_versions.min),
            (select version from appversions where
                id=applications_versions.max)
        FROM addons
            INNER JOIN versions ON versions.addon_id=addons.id
            INNER JOIN licenses ON licenses.id=versions.license_id
            INNER JOIN files ON files.version_id=versions.id
            INNER JOIN applications_versions ON
                applications_versions.version_id=versions.id
        WHERE addons.status=4
        """)
    for row in rows:
        addon_id, bundle_id, license_id, alicense, release_date, version, \
                releasenotes, filename, sugar_min, sugar_max = row
        if [i for i in EXCLUDE_BUNDLE_IDS if i in bundle_id]:
            continue

        if license_id == 0:
            alicense = 'MPLv1.1'
        elif license_id == 1:
            alicense = 'GPLv2'
        elif license_id == 2:
            alicense = 'GPLv3'
        elif license_id == 3:
            alicense = 'LGPLv2'
        elif license_id == 4:
            alicense = 'LGPLv3'
        elif license_id == 5:
            alicense = 'MIT'
        elif license_id == 6:
            alicense = 'BSD'
        else:
            for good in server.GOOD_LICENSES:
                if not alicense or good in ['ec']:
                    continue
                if good in alicense:
                    alicense = good
                    break
            else:
                for words, good in MISNAMED_LICENSES.items():
                    for i in words:
                        if i not in alicense.lower():
                            break
                    else:
                        alicense = good
                        break
                else:
                    print '-- Skip bad license %r' % alicense
                    continue

        for i in client.Context.find(implement=[bundle_id]):
            if i['implement'] == [bundle_id]:
                context_guid = i['guid']
                break
        else:
            context_guid = context_new(addon_id, bundle_id)

        impl = client.Implementation()
        impl['context'] = context_guid
        impl['license'] = [alicense]
        impl['version'] = version
        impl['date'] = int(time.mktime(release_date.timetuple()))
        impl['stability'] = 'stable'
        impl['notes'] = releasenotes or ''
        impl['feed'] = {
                'requires': {
                    sugar_guid: {'constraints': sugar_max},
                    },
                }
        impl.post()
        impl.set_blob_with_url('bundle',
                'http://download.sugarlabs.org/activities/%s/%s' % \
                        (addon_id, filename))


def context_new(addon_id, bundle_id):
    context = client.Context()
    context['type'] = 'activity'
    context['implement'] = [bundle_id]

    context['title'], context['summary'], context['description'], \
            context['homepage'], icondata = sqlexec("""
        SELECT
            (select max(localized_string) from translations where
                id=addons.name),
            (select max(localized_string) from translations where
                id=addons.summary),
            (select max(localized_string) from translations where
                id=addons.description),
            (select max(localized_string) from translations where
                id=addons.homepage),
            icondata
        FROM addons
        WHERE id=%s
        """ % addon_id)[0]
    context['summary'] = context['summary'] or context['title']
    context['description'] = context['description'] or context['title']

    tags = set()
    for row in sqlexec("""
            SELECT
                (select localized_string from translations where
                    id=categories.name AND locale='en-US')
            FROM addons_categories
                INNER JOIN categories ON
                    categories.id=addons_categories.category_id
            WHERE
                addons_categories.addon_id=%s
            """ % addon_id):
        tags.add(CATEGIORIES_TO_TAGS[row[0]])
    for row in sqlexec("""
            SELECT
                tags.tag_text
            FROM users_tags_addons
                INNER JOIN tags ON tags.id=users_tags_addons.tag_id
                INNER JOIN addons_users ON
                    addons_users.addon_id=users_tags_addons.addon_id
            WHERE
                users_tags_addons.addon_id=%s
            """ % addon_id):
        tags.add(row[0])
    context['tags'] = list(tags)

    context.post()

    if icondata:
        context.set_blob('icon', icondata)

    for row in sqlexec("""
            SELECT
                (select max(localized_string) from translations where
                    id=caption),
                thumbdata,
                filedata
            FROM previews
            WHERE
                addon_id=%s
            """ % addon_id):
        caption, thumb, data = row

        artifact = client.Artifact()
        artifact['context'] = context['guid']
        artifact['type'] = 'screenshot'
        artifact['title'] = caption or ''
        artifact['description'] = caption or ''
        artifact.post()

        if thumb:
            artifact.set_blob('preview', thumb)
        if data:
            artifact.set_blob('data', data)

    return context['guid']


def sqlexec(text):
    cur = connection.cursor()
    cur.execute(text)
    return cur.fetchall()


if __name__ == '__main__':
    server.debug.value = 3
    server.data_root.value = 'tmp/db'
    server.stats_root.value = 'tmp/stats'
    server.logdir.value = 'tmp/log'
    server.index_write_queue.value = 1024 * 10
    server.index_flush_threshold.value = 0
    server.index_flush_timeout.value = 0

    server_pid = restful_document.fork(server.resources())

    client.api_url.value = \
            'http://%s:%s' % (server.host.value, server.port.value)
    try:
        main()
    finally:
        os.kill(server_pid, signal.SIGINT)
        os.waitpid(server_pid, 0)
