#!/usr/bin/python
"""
This file is part of the 'Elements' Project
Elements is a 2D Physics API for Python (supporting Box2D2)

Copyright (C) 2008, The Elements Team, <elements@linuxuser.at>

Home:  http://wiki.laptop.org/go/Elements
IRC:   #elements on irc.freenode.org

Code:  http://www.assembla.com/wiki/show/elements
       svn co http://svn2.assembla.com/svn/elements                     

License:  GPLv3 | See LICENSE for the full text
This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.              
"""

from math import fabs, sqrt, atan, degrees
from random import shuffle

import tools_poly
import drawing

# Part for loading the appropriate lib for the current platform
from platform import architecture
from platform import system as platformsystem

s = platformsystem()
arch, arch2 = architecture()
print "Loading box2d for %s (%s)" % (s, arch)

if s == 'Linux':
    if arch == "64bit": 
        import box2d_linux64 as box2d
    else: 
        try:
            import box2d_linux32 as box2d
        except:
            import box2d_linux32ppc as box2d

elif s == 'Windows': 
    import Box2D2 as box2d 
    # Windows version doesn't follow the naming scheme, it's just as released by kne

elif s == 'Darwin': 
    import box2d_macosx as box2d
        
# Main Class
class Elements:
    """The class which handles all interaction with the box2d engine
    """
    run_physics = True   # Can pause the simulation
    element_count = 0    # Element Count 
    elements = []        # Elements reference
    points = []          # Debug poly drawing points
    drawer = None        # Specific drawing class
    
    def __init__(self, screen_size, gravity=(0.0,-9.0), ppm=100.0):
        """Init the world with boundaries and gravity, and init colors.
        Returns: class Elements
        
        Parameters:
          screen_size -- (w, h) .. screen size in pixels
          gravity -- m/s^2 (x, y)
          ppm     -- pixels per meter
        """
        
        self.display_width, self.display_height = screen_size
        
        # Set Boundaries
        self.worldAABB=box2d.b2AABB()
        self.worldAABB.lowerBound.Set(-100.0, -100.0)
        self.worldAABB.upperBound.Set(100.0, 100.0)
        
        # Gravity + Bodies will sleep on outside
        gx, gy = gravity
        self.gravity = box2d.b2Vec2(gx, gy);
        self.doSleep = True
    
        # Create the World
        self.world = box2d.b2World(self.worldAABB, self.gravity, self.doSleep)

        # Get display size        
        self.init_colors()
        
        # Set Pixels per Meter
#        self.ppm = self.display_width / 4.
        self.ppm = ppm

    def get_elementCount(self):
        return self.element_count

    def set_drawingMethod(self, m):
        if m == "pygame":
            self.drawer = drawing.draw_pygame()
            
    def set_screenSize(self, size):
        """ Get the current screen size
            Parameters: size == (int(width), int(height)) 
        """
        self.display_width, self.display_height = size

    def init_colors(self):
        """ Init self.colors with a fix set of hex colors 
        """
        self.fixed_color = None
        self.cur_color = 0
        self.colors = [
          "#737934", "#729a55", "#040404", "#1d4e29", "#ae5004", "#615c57",
          "#6795ce", "#203d61", "#8f932b"
        ]
        shuffle(self.colors)

    def set_color(self, clr):
        """ Set a color for all future Elements, until reset_color() is called 
            Parameter: clr == (Hex or RGB)
        """
        self.fixed_color = clr
    
    def reset_color(self):
        """ All Elements from now on will be drawn in random colors 
        """
        self.fixed_color = None

    def get_color(self):
        """ Get a color - either the fixed one or the next from self.colors 
            Returns: clr = ((R), (G), (B)) 
        """
        if self.fixed_color != None:
            return self.fixed_color
            
        if self.cur_color == len(self.colors): 
            self.cur_color = 0
            shuffle(self.colors)
    
        clr = self.colors[self.cur_color]
        if clr[0] == "#":
            clr = tools_poly.hex2rgb(clr)
        
        self.cur_color += 1
        return clr
    
    def update(self, fps=50.0, iterations=10):
        if not self.run_physics: return
        
        timeStep = 1.0 / fps
        self.world.Step(timeStep, iterations);
    
    def draw(self, surface):
#        body = self.world.GetBodyList()        
        if not self.drawer: # No need to run through the loop if there's no way to draw!
            return
            
        for element in self.elements:
            clr = element.userData['color']
            
            xform = element.GetXForm()
            shape = element.GetShapeList()
            
            while shape:                
                type = shape.GetType()
                
                if type == box2d.e_circleShape:
                    circle = shape.asCircle()
                    position = box2d.b2Mul(xform, circle.GetLocalPosition())
                    rect = ((position.x-circle.m_radius)*self.ppm,
                      self.display_height - (position.y+circle.m_radius)*self.ppm,
                      (2*circle.m_radius)*self.ppm,
                      (2*circle.m_radius)*self.ppm)
                    self.drawer.draw_ellipse(surface, clr, rect, 2)

                elif type == box2d.e_polygonShape:
                    poly = shape.asPolygon()
                    points = []
                    for i in range(0,poly.GetVertexCount()):
                        pt = box2d.b2Mul(xform, poly.getVertex(i))
                        pt.x = pt.x * self.ppm
                        pt.y = self.display_height - (pt.y * self.ppm)
                        points.append([pt.x, pt.y])

                    self.drawer.draw_polygon(surface, clr, points, 2)
                   
                else:
                    print "  unknown shape type:%d" % shape.GetType()
    
                shape = shape.GetNext()  
        
    def add_wall(self, a, b):
        """ add a wall (static rectangle) with corners (ax, ay)-(bx, by) where bx, by > ax, ay
            all in pixels"""
        ax, ay = a
        ax /= self.ppm
        ay /= self.ppm
        
        bx, by = b
        bx /= self.ppm
        by /= self.ppm

#        print ax,ay,"--",bx,by
        BodyDef = box2d.b2BodyDef()
        BodyDef.position.Set(ax, ay)

        Body = self.world.CreateStaticBody(BodyDef)
        userData = { 'color' : self.get_color() }
        Body.userData = userData
        self.elements.append(Body)
        self.element_count = len(self.elements)        

        ShapeDef = box2d.b2PolygonDef()
        ShapeDef.SetAsBox(max(0.1,bx-ax), max(0.1,by-ay))
        Body.CreateShape(ShapeDef)
        Body.SetMassFromShapes()
    
    def add_ground(self, pos=(0.0, -10.0)):
        """add a very large static rectangle at pos (x, y) in meters,
           mainly so objects don't fall below the playing field"""
        # Define the body
        BodyDef = box2d.b2BodyDef()
        BodyDef.position.Set(pos[0], pos[1])
        
        # Create the Body
        body = self.world.CreateStaticBody(BodyDef)
        userData = { 'color' : self.get_color() }
        body.userData = userData
        self.elements.append(body)
        self.element_count = len(self.elements)
        
        # Add a shape to the Body
        ShapeDef = box2d.b2PolygonDef()
        ShapeDef.SetAsBox(50.0, 10.0)
        ShapeDef.userData = self.get_color()

        body.CreateShape(ShapeDef)
        body.SetMassFromShapes()

    
    def add_ball(self, pos, radius=0.5, density=1.0, restitution=0.16, friction=0.5):
        """add a dynamic ball at pos (in pixels), with optional radius, density, restitution, friction"""
        x, y = pos
        x /= self.ppm
        y = self.display_height - y
        y /= self.ppm
        
        # Define the body
        bodyDef = box2d.b2BodyDef()
        bodyDef.position.Set(x, y)

        # Create the Body
        body = self.world.CreateDynamicBody(bodyDef)
        userData = { 'color' : self.get_color() }
        body.userData = userData
        
        self.elements.append(body)
        self.element_count = len(self.elements)

        # Add a shape to the Body
        circleDef = box2d.b2CircleDef()
        circleDef.density = density
        circleDef.radius = radius
        circleDef.restitution = restitution
        circleDef.friction = friction

        body.CreateShape(circleDef)
        body.SetMassFromShapes();        

    def add_rect(self, pos, a=1, b=0.5, density=1.0, restitution=0.16, friction=0.5):
        """add a dynamic rectangle at pos (x, y) in pixels, width of a, height of b (meters)"""
        x, y = pos
        x /= self.ppm
        y = self.display_height - y
        y /= self.ppm
        
        # Define the body
        bodyDef = box2d.b2BodyDef()
        bodyDef.position.Set(x, y)

        # Create the Body
        body = self.world.CreateDynamicBody(bodyDef)
        userData = { 'color' : self.get_color() }
        body.userData = userData
        
        self.elements.append(body)
        self.element_count = len(self.elements)

        # Add a shape to the Body
        boxDef = box2d.b2PolygonDef()
        boxDef.SetAsBox(a, b)
        boxDef.density = density
        boxDef.restitution = restitution
        boxDef.friction = friction
        body.CreateShape(boxDef)
        
        body.SetMassFromShapes();

    def add_triangle(self, pos, density=1.0, restitution=0.16, friction=0.5):
        """add a triangle (local vertices (-1,0) (1,0) (0,2) in meters) at pos (x, y) in pixels"""
        x, y = pos
        x /= self.ppm
        y = self.display_height - y
        y /= self.ppm
        
        # Define the body
        bodyDef = box2d.b2BodyDef()
        bodyDef.position.Set(x, y)

        # Create the Body
        body = self.world.CreateDynamicBody(bodyDef)
        userData = { 'color' : self.get_color() }
        body.userData = userData
        self.elements.append(body)
        self.element_count = len(self.elements)

        # Add a shape to the Body
        polyDef = box2d.b2PolygonDef()
        polyDef.vertexCount = 3
        polyDef.setVertex(0, box2d.b2Vec2(-1.0, 0.0))
        polyDef.setVertex(1, box2d.b2Vec2(1.0, 0.0))
        polyDef.setVertex(2, box2d.b2Vec2(0.0, 2.0))        
        
        polyDef.density = density
        polyDef.restitution = restitution
        polyDef.friction = friction
        body.CreateShape(polyDef)
        
        body.SetMassFromShapes();

    def add_poly(self, pos, vertices, density=1.0, restitution=0.16, friction=0.5):
        """ add a polygon at pos (x, y) in meters. vertices as a list or tuple.

        NOTE: Box2D has a maximum poly vertex count, defined in Common/box2d.b2Settings.h (box2d.b2_maxPolygonVertices)
        We need to make sure, that we reach that by reducing the poly with increased tolerance
        """
        print

        # Reduce Polygon
        tolerance = 20
        v_new = None
        while v_new == None or len(v_new) > 16:
            tolerance += 3    
            v_new = tools_poly.reduce_poly(vertices, tolerance)

        vertices = v_new
        print "Polygon reduced to %i vertices | tolerance: %i" % (len(vertices), tolerance)
        
        """ Problem: Straight lines will crash the Box2D engine.
        How to detect 'straight' lines. We try by comparing the vectors. (tools_poly.py)
        """
        if tools_poly.is_line(vertices):
            print "IS LINE"
            x1, y1 = vertices[0]
            x2, y2 = vertices[-1]
            
            vx, vy = vector = (x2-x1, y2-y1)
            l = sqrt((vx*vx) + (vy*vy))
            
            # Make a normalized Vector (for a hypothenusis with length=1)
            if l == 0.0:
                # Extremely short line
                self.add_rect(pos, 0.07, 0.07)
                return

            vx /= l
            vy /= l
            
            factor = 20
            vnx, vny = v_norm = (-vy*factor, vx*factor)
            
            p1 = (x1+vnx, y1+vny)
            p2 = (x2+vnx, y2+vny)
            
            vertices = []
            vertices.append((x1, y1))
            vertices.append((x2, y2))
            vertices.append(p2)
            vertices.append(p1)
            print "normal vector = (%.4f, %.4f)" % (vnx, vny)            
            print "%i vertices" % (len(vertices))
             
        # So poly should be alright
        # Continue reducing the vertecs
        vertices_orig_reduced = vertices    
        vertices = tools_poly.poly_center_vertices(vertices)
        vertices = tools_poly.convex_hull(vertices)

        if len(vertices) > box2d.b2_maxPolygonVertices  or len(vertices) < 3: 
            return

        # Define the body
        x, y = c = tools_poly.calc_center(vertices_orig_reduced)
        x /= self.ppm
        y = self.display_height - y
        y /= self.ppm

        bodyDef = box2d.b2BodyDef()
        bodyDef.position.Set(x, y)

        # Create the Body
        body = self.world.CreateDynamicBody(bodyDef)
        userData = { 'color' : self.get_color() }
        body.userData = userData
        
        self.elements.append(body)
        self.element_count = len(self.elements)

        # Add a shape to the Body
        polyDef = box2d.b2PolygonDef()
        polyDef.vertexCount = len(vertices)
        for i in range(len(vertices)):
            vx, vy = vertices[i]
            vx /= self.ppm
            vy /= self.ppm
#            print i, vx, vy
            polyDef.setVertex(i, box2d.b2Vec2(vx, vy))
        
        
        polyDef.density = density
        polyDef.restitution = restitution
        polyDef.friction = friction
        body.CreateShape(polyDef)
        
        body.SetMassFromShapes()

