"""
  Elements - 2D rigid body physics for python (supporting box2d and chipmunk)
 
  Copyright (C) 2008, Chris Hager and Joshua Minor

        Home:  http://wiki.laptop.org/go/Elements
         IRC:  #elements on irc.freenode.org
         
        Code:  http://www.assembla.com/wiki/show/elements
               svn co http://svn2.assembla.com/svn/elements
  
  This file is part of the Elements Project
  
      Elements is free software: you can redistribute it and/or modify
      it under the terms of the GNU General Public License as published by
      the Free Software Foundation, either version 3 of the License, or
      (at your option) any later version.
  
      Elements is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      GNU General Public License for more details.
  
      You should have received a copy of the GNU General Public License
      along with Elements.  If not, see <http://www.gnu.org/licenses/>.
"""

import pygame
from pygame.locals import *
from pygame.color import *

from time import sleep
from random import shuffle

# Part for loading the appropriate lib for the current platform
from platform import architecture
from platform import system as platformsystem

s = platformsystem()
arch, arch2 = architecture()
print "Loading box2d for %s (%s)" % (s, arch)

if s == 'Linux':
    if arch == "64bit": 
        from box2d_linux64 import *
    else: 
        from box2d_linux32 import *        

elif s == 'Windows': 
    from box2d_windows import *        

elif s == 'Darwin': 
    from box2d_macosx import *

from threading import Thread

class ShowFPS(Thread):
	count = 0
	def __init__(self, seconds, pygame_clock):
		Thread.__init__(self)
		self.running = True
		self.clock = pygame_clock
		self.delay = seconds
		
	def run(self):
		i = 0
		m = 5.0
		while self.running:
			sleep(self.delay / m)
			i += 1
			if i == m:
				print "Elements: %i \t %i FPS" % (self.count-3, int(self.clock.get_fps()))
				i = 0

# Some Hex Tools
def hex2dec(hex): return int(hex, 16)
def hex2rgb(hex): 
    if hex[0:1] == '#': hex = hex[1:]; 
    return (hex2dec(hex[:2]), hex2dec(hex[2:4]), hex2dec(hex[4:6]))

# Element Wrapper, as box2d's userData seems not to work with python,
# and we want to attach specific information to bodies
class element:
    def __init__(self, body, userData=''):
        self.body = body
        self.userData = userData
        
# Main Class
class elements:
    run_physics = True
    element_count = 0
    elements = []
    
    def __init__(self, gravity=(0.0,-9.0), ppm=100.0):
        # Set Boundaries
        self.worldAABB=b2AABB()
        self.worldAABB.lowerBound.Set(-100.0, -100.0)
        self.worldAABB.upperBound.Set(100.0, 100.0)
        
        # Gravity + Bodies will sleep on outside
        gx, gy = gravity
        self.gravity = b2Vec2(gx, gy);
        self.doSleep = True
    
        # Create the World
        self.world = b2World(self.worldAABB, self.gravity, self.doSleep)

        # Get display size        
        self.autoset_screen_size()
#        self.init_colors()
        self.set_color((0,0,0))
        
        # Set Pixels per Meter
#        self.ppm = self.display_width / 4.
        self.ppm = ppm

    def autoset_screen_size(self, size=None):
        """ Get the current PyGame Screen Size, or sets it manually
            Optional: size == (int(width), int(height)) 
        """
        if size != None:
            self.display_width, self.display_height = size
            return
            
        try:
            x,y,w,h = pygame.display.get_surface().get_rect()
            self.display_width = w
            self.display_height = h
        except:
            print "Elements Error: Please start pygame.init() before loading Elements Physics"
            exit(0)

    def init_colors(self):
        """ Init self.colors with a fix set of hex colors 
        """
        self.fixed_color = None
        self.cur_color = 0
        self.colors = [
          "#737934", "#729a55", "#040404", "#1d4e29", "#ae5004", "#615c57",
          "#6795ce", "#203d61", "#8f932b"
        ]
        shuffle(self.colors)

    def set_color(self, clr):
        """ Set a color for all future Elements, until reset_color() is called 
            Parameter: clr == (Hex or RGB)
        """
        self.fixed_color = clr
    
    def reset_color(self):
        """ All Elements from now on will be drawn in random colors 
        """
        self.fixed_color = None

    def get_color(self):
        """ Get a color - either the fixed one or the next from self.colors 
            Returns: clr = ((R), (G), (B)) 
        """
        if self.fixed_color != None:
            return self.fixed_color
            
        if self.cur_color == len(self.colors): 
            self.cur_color = 0
            shuffle(self.colors)
    
        clr = self.colors[self.cur_color]
        if clr[:1] == "#":
            clr = hex2rgb(clr)
        
        self.cur_color += 1
        return clr

    def update(self, fps=40.0, iterations=10):
#    	print fps, iterations
        timeStep = 1.0 / fps
        self.world.Step(timeStep, iterations);
    
    def draw(self, surface):
#        body = self.world.GetBodyList()
        
        for element in self.elements:
            body = element.body
            clr = element.userData['color']
#            clr = (0,0,0)
            
            xform = body.GetXForm()
            shape = body.GetShapeList()
            
            while shape:
                i = shape.GetUserData()
                
                type = shape.GetType()
                if type == e_circleShape:
                    circle = shape.asCircle()
                    position = b2Mul(xform, circle.GetLocalPosition())
                    rect = pygame.Rect((position.x-circle.m_radius)*self.ppm,
                      self.display_height - (position.y+circle.m_radius)*self.ppm,
                      (2*circle.m_radius)*self.ppm,
                      (2*circle.m_radius)*self.ppm)
                    pygame.draw.ellipse(surface, clr, rect, 2)
                    
                elif type == e_polygonShape:
                    poly = shape.asPolygon()
                    points = []
                    for i in range(0,poly.GetVertexCount()):
                        pt = b2Mul(xform, poly.getVertex(i))
                        pt.x = pt.x * self.ppm
                        pt.y = self.display_height - (pt.y * self.ppm)
                        points.append([pt.x, pt.y])
                    pygame.draw.polygon(surface, clr, points, 2)
    
                else:
                    print "  unknown shape type:%d" % shape.GetType()
    
                shape = shape.GetNext()            
#            body = body.GetNext()
        
    def add_wall(self, a, b):
        ax, ay = a
        ax /= self.ppm
        ay /= self.ppm
        
        bx, by = b
        bx /= self.ppm
        by /= self.ppm

#        print ax,ay,"--",bx,by
        BodyDef = b2BodyDef()
        BodyDef.position.Set(ax, ay)

        Body = self.world.CreateStaticBody(BodyDef)
        userData = { 'color' : self.get_color() }
        self.elements.append(element(Body, userData))
        self.element_count = len(self.elements)        

        ShapeDef = b2PolygonDef()
        ShapeDef.SetAsBox(max(0.1,bx-ax), max(0.1,by-ay))
        Body.CreateShape(ShapeDef)
        Body.SetMassFromShapes()
    
    def add_ground(self):
        # Define the body
        BodyDef = b2BodyDef()
        BodyDef.position.Set(0.0, -10.0)
        
        # Create the Body
        body = self.world.CreateStaticBody(BodyDef)
        userData = { 'color' : self.get_color() }
        self.elements.append(element(body, userData))
        self.element_count = len(self.elements)
        
        # Add a shape to the Body
        ShapeDef = b2PolygonDef()
        ShapeDef.SetAsBox(50.0, 10.0)
        ShapeDef.userData = self.get_color()

        body.CreateShape(ShapeDef)
        body.SetMassFromShapes()

    
    def add_ball(self, pos):
        x, y = pos
        x /= self.ppm
        y = self.display_height - y
        y /= self.ppm
        
        # Define the body
        bodyDef = b2BodyDef()
        bodyDef.position.Set(x, y)

        # Create the Body
        body = self.world.CreateDynamicBody(bodyDef)

        userData = { 'color' : self.get_color() }
        self.elements.append(element(body, userData))
        self.element_count = len(self.elements)

        # Add a shape to the Body
        circleDef = b2CircleDef()
        circleDef.density = 1.0
        circleDef.radius = 0.1
        circleDef.restitution = 0.15
        circleDef.friction = 0.5

        body.CreateShape(circleDef)
        body.SetMassFromShapes();        

    def add_rect(self, pos, a=0.1, b=0.1):
        x, y = pos
        x /= self.ppm
        y = self.display_height - y
        y /= self.ppm
        
        # Define the body
        bodyDef = b2BodyDef()
        bodyDef.position.Set(x, y)

        # Create the Body
        body = self.world.CreateDynamicBody(bodyDef)

        userData = { 'color' : self.get_color() }
        self.elements.append(element(body, userData))
        self.element_count = len(self.elements)

        # Add a shape to the Body
        boxDef = b2PolygonDef()
        boxDef.SetAsBox(a, b)
        boxDef.density = 1.0
        boxDef.restitution = 0.15
        boxDef.friction = 0.5

        body.CreateShape(boxDef)
        body.SetMassFromShapes();
        