/* activity_widgets.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Tool button that represnts activity
 *
 * Button shows icon, color and tooltip of an activity.
 *
 * If activity has many tool items, use ActivityToolbarButton to save space
 * in toolbar by placing all activity related items to ActivityToolbarButton's
 * sub-toolbar.
 *
 * It is a good practice to have this button at leftmost position in a toolbar.
 */
public class Sugar.ActivityButton : Sugar.ToolButton {
    /**
     * @param jobject   activity jobject to be based on
     */
    public ActivityButton (Jobject jobject) {
        _jobject = jobject;

        icon_name = Environ.activity.icon;
        icon.xo_color = _jobject.icon_color;

        tooltip = _jobject.title;
        _jobject.notify["title"].connect (_notify_title_cb);
    }

    private void _notify_title_cb (Object sender, ParamSpec param) {
        tooltip = _jobject.title;
    }

    private Jobject _jobject;
}

/**
 * Tool button and sub-toolbar that represnt activity
 *
 * Button shows icon, color of an activity. Sub-toolbar provides default set
 * of activity related widgets thus it is a good method to save space in toolbar
 * for other activity tool items.
 *
 * It is a good practice to have this button at leftmost position in a toolbar.
 */
public class Sugar.ActivityToolbarButton : Sugar.ToolbarButton {
    /**
     * Activity title widget
     */
    public TitleEntry title { get; private set; }

    /**
     * Activity keep button
     */
    public KeepButton keep { get; private set; }

    /**
     * @param activity  activity to create button for
     */
    public ActivityToolbarButton (Activity activity) {
        icon_name = Environ.activity.icon;

        var toolbar = new Gtk.Toolbar ();
        page = toolbar;

        if (activity.jobject == null)
            icon.xo_color = Profile.color;
        else {
            icon.xo_color = activity.jobject.icon_color;

            title = new TitleEntry (activity.jobject);
            toolbar.insert (title, -1);

            keep = new KeepButton (activity.jobject);
            toolbar.insert (keep, -1);
        }

        if (activity.connection != null) {
            var share = new ShareButton (activity.connection);
            toolbar.insert (share, 1);
        }

        toolbar.show_all ();
    }
}

/**
 * Tool button to close activity
 *
 * Class just presets defualt values for properties like icon and tooltip,
 * connect to clicked event to make button useful.
 *
 * It is a good practice to have this button at rightmost position in a toolbar
 * prepended by expanded separator.
 */
public class Sugar.StopButton : Sugar.ToolButton {
    construct {
        icon_name = "activity-stop";
        tooltip = _("Stop");
        accelerator = "<Ctrl>Q";
    }
}

/**
 * Tool button to undo changes
 *
 * Class just presets defualt values for properties like icon and tooltip,
 * connect to clicked event to make button useful.
 */
public class Sugar.UndoButton : Sugar.ToolButton {
    construct {
        icon_name = "edit-undo";
        tooltip = _("Undo");
        accelerator = "<Ctrl>Z";
    }
}

/**
 * Tool button to redo changes
 *
 * Class just presets defualt values for properties like icon and tooltip,
 * connect to clicked event to make button useful.
 */
public class Sugar.RedoButton : Sugar.ToolButton {
    construct {
        icon_name = "edit-redo";
        tooltip = _("Redo");
    }
}

/**
 * Tool button to copy
 *
 * Class just presets defualt values for properties like icon and tooltip,
 * connect to clicked event to make button useful.
 */
public class Sugar.CopyButton : Sugar.ToolButton {
    construct {
        icon_name = "edit-copy";
        tooltip = _("Copy");
    }
}

/**
 * Tool button to paste
 *
 * Class just presets defualt values for properties like icon and tooltip,
 * connect to clicked event to make button useful.
 */
public class Sugar.PasteButton : Sugar.ToolButton {
    construct {
        icon_name = "edit-paste";
        tooltip = _("Paste");
    }
}

/**
 * Tool button to save a copy of journal entry
 *
 * On click event, button will save a copy of passed Jobject.
 */
public class Sugar.KeepButton : Sugar.ToolButton {
    /**
     * @param jobject   activity jobject to be based on
     */
    public KeepButton (Jobject jobject) {
        _jobject = jobject;

        tooltip = _("Keep");
        accelerator = "<Ctrl>S";

        icon_name = "document-save";
        icon.xo_color = Profile.color;

        clicked.connect (_clicked_cb);
    }

    private void _clicked_cb () {
        _jobject.cow ();
    }

    private Jobject _jobject;
}

/**
 * Tool button to change activity sharing status
 *
 * After choosing entry from button palette, it will set Connection.status
 * property value.
 */
public class Sugar.ShareButton : Sugar.ComboToolButton {
    /**
     * @param connection    activity connection to be based on
     */
    public ShareButton (Connection connection) {
        var palette = new Palette ();
        palette.primary_text = _("Share scope");
        invoker.palette = palette;

        var menu = new Menu ();
        palette.secondary = menu;

        action = new Gtk.RadioAction ("", _("Private"), null, null, 0);
        action.icon_name = "zoom-home";
        menu.insert_action (action);

        var @public = new Gtk.RadioAction ("", _("My Neighborhood"),
                null, null, 1);
        @public.icon_name = "zoom-neighborhood";
        @public.set_group (action.get_group ());
        menu.insert_action (@public);

        menu.show_all ();

        _connection = connection;
        _connection.scope_changed.connect (_scope_changed_cb);
        _connection.notify["scope-changing"].connect (_scope_changing_cb);

        action.current_value = _connection.scope == ShareScope.PRIVATE ? 0 : 1;
        action.changed.connect (_changed_cb);
    }

    private void _scope_changed_cb (ShareScope prev_state, string? error) {
        if (error != null)
            AlertBin.push (new NotifyAlert (_("Sharing error"), error,
                            "emblem-warning"));
        action.current_value = _connection.scope == ShareScope.PRIVATE ? 0 : 1;
    }

    private void _scope_changing_cb (Object sender, ParamSpec param) {
        action.sensitive = !_connection.scope_changing;
    }

    private void _changed_cb (Gtk.RadioAction current) {
        _connection.scope = current.current_value == 0
                ? ShareScope.PRIVATE : ShareScope.PUBLIC;
    }

    private Connection _connection;
}

/**
 * Tool item to change activity title
 *
 * Widget will change Jobject.title implicitly. Also it will reflect on Jobject
 * updates.
 */
public class Sugar.TitleEntry : Gtk.ToolItem {
    /**
     * Entry widget which represnts title
     */
    public Gtk.Entry entry { get; private set; }

    /**
     * @param jobject   activity jobject to be based on
     */
    public TitleEntry (Jobject jobject) {
        _jobject = jobject;

        entry = new Gtk.Entry ();
        entry.set_size_request (Gdk.Screen.width () / 3, -1);
        entry.text = _jobject.title;
        entry.changed.connect (_entry_changed_cb);
        entry.show ();
        add (entry);

        _jobject.notify["title"].connect (_notify_title_cb);
    }

    private void _notify_title_cb (Object sender, ParamSpec param) {
        entry.text = _jobject.title;
    }

    private void _entry_changed_cb () {
        if (_timeout != 0)
            Source.remove (_timeout);
        _timeout = Timeout.add_seconds (_TIMEOUT_IN_SEC, _timeout_cb);
    }

    private bool _timeout_cb () {
        _timeout = 0;

        _jobject.title = entry.text;
        _jobject.title_set_by_user = true;
        _jobject.write ();

        return false;
    }

    private const int _TIMEOUT_IN_SEC = 1;

    private Jobject _jobject;
    private uint _timeout;
}
