/* profile.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Current user profile
 */
public class Sugar.Profile {
    /**
     * Name of sugar profile
     *
     * This value is equal to profile sub-directory name in ~/.sugar.
     */
    public static string name {
        get {
            var @value = Environment.get_variable ("SUGAR_PROFILE");
            return @value == null ? "default" : @value;
        }
    }

    /**
     * Path to sugar profile
     *
     * Full path to profile sub-directory in ~/.sugar.
     */
    public static string path {
        owned get {
            return Path.build_filename (Environment.get_home_dir (),
                    Environ.ROOT_DIRECTORY, name);
        }
    }

    /**
     * Pair of user's own colors
     */
    public static XoColor color {
        owned get {
            string? colors = null;

            try {
                colors = Config.instance ().get_color ();
            } catch (Error e) {
                warning ("Cannot get color profile info: %s", e.message);
            }

            if (colors == null) {
                warning ("User color is empty, will use default");
                return _default_color;
            }

            var xo_color = XoColor.from_string (colors);

            if (xo_color == null) {
                warning ("Cannot parse user color, will use default");
                return _default_color;
            }

            return xo_color;
        }
    }

    /**
     * User's short name
     */
    public static string nickname {
        owned get {
            try {
                return Config.instance ().get_nickname ();
            } catch (Error e) {
                warning ("Cannot get nickname profile info: %s", e.message);
                return "";
            }
        }
    }

    private Profile () {
    }

    private const XoColor _default_color = {
        { 1.0, 0x0000, 0x0000, 0x0000 },
        { 1.0, 0xFFFF, 0xFFFF, 0xFFFF }
    };
}
