/* accelerator.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

public class Sugar.Accelerator {
    public Accelerator (Gtk.Widget accel_widget, string signal_name) {
        _accel_widget = accel_widget;
        _signal_name = signal_name;

        _accel_widget.can_activate_accel.connect (_can_activate_accel_cb);
        _accel_widget.hierarchy_changed.connect (_hierarchy_changed_cb);
    }

    ~Accelerator () {
        _accel_widget.can_activate_accel.disconnect (_can_activate_accel_cb);
        _accel_widget.hierarchy_changed.disconnect (_hierarchy_changed_cb);
    }

    public string? name {
        get {
            return _name;
        }
        set {
            if (value == name)
                return;

            if (name != null) {
                _accelerator_remove (_accel_widget.get_toplevel ());
                _name = null;
            }

            if (value != null) {
                Gtk.accelerator_parse (value, out _key, out _mods);
                if (_key == 0) {
                    warning (@"Bad accelerator value: $(@value)");
                    return;
                }
            }

            _name = value;

            if (name != null)
                _accelerator_add ();
        }
    }

    private void _accelerator_add () {
        var toplevel = _accel_widget.get_toplevel ();
        if (toplevel == null || !toplevel.is_toplevel ())
            return;

        Gtk.AccelGroup accel_group = toplevel.get_data ("sugar-accel-group");
        if (accel_group == null) {
            warning (@"No Gtk.AccelGroup in $(toplevel.get_type ().name ())");
            return;
        }

        _accel_widget.add_accelerator (_signal_name, accel_group, _key, _mods,
                Gtk.AccelFlags.LOCKED | Gtk.AccelFlags.VISIBLE);
    }

    private void _accelerator_remove (Gtk.Widget? previous_toplevel) {
        if (previous_toplevel == null || !previous_toplevel.is_toplevel ())
            return;

        Gtk.AccelGroup accel_group = previous_toplevel.get_data (
                "sugar-accel-group");
        if (accel_group != null)
            _accel_widget.remove_accelerator (accel_group, _key, _mods);
    }

    private bool _can_activate_accel_cb (uint signal_id) {
        // Accept activation via accelerators regardless of
        // widget's state
        return true;
    }

    private void _hierarchy_changed_cb (Gtk.Widget accel_widget,
            Gtk.Widget? previous_toplevel) {
        if (name != null) {
            _accelerator_remove (previous_toplevel);
            _accelerator_add ();
        }
    }

    private Gtk.Widget _accel_widget;
    private string _signal_name;
    private string? _name;
    private uint _key;
    private Gdk.ModifierType _mods;
}
