/* icon.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Port from original sugar-toolkit project.
 * File:   src/sugar/graphics/icon.py
 * Commit: 4a4ea538cacd740f8e32f625d400e26ab590daa0
 *
 * Copyright (C) 2006-2007 Red Hat, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

using Gee;

public class Sugar.Icon : Gtk.Image {
    public new string? file {
        get {
            return _file;
        }
        set {
            if (value == file)
                return;

            _file = value;

            if (value[0] == '/')
                _attr.file_name = file;
            else
                _attr.icon_name = file;

            queue_draw ();
        }
    }

    public Color fill_color {
        get { return _attr.fill_color; }
        set { _attr.fill_color = value; queue_draw (); }
    }

    public Color stroke_color {
        get { return _attr.stroke_color; }
        set { _attr.stroke_color = value; queue_draw (); }
    }

    public XoColor xo_color {
        set { _attr.xo_color = value; }
    }

    public int width {
        get { return _attr.width; }
    }

    public int height {
        get { return _attr.height; }
    }

    public new int pixel_size {
        set {
            _attr.width = _attr.height = value;
            queue_draw ();
        }
    }

    public new Gtk.IconSize icon_size {
        set {
            _attr.icon_size = value;
            queue_draw ();
        }
    }

    public bool pulsing {
        get {
            return _pulse_hid != 0;
        }
        set {
            if (value == pulsing)
                return;
            if (value) {
                _pulse_phase = 0.0;
                _pulse_hid = Timeout.add (_PULSE_INTERVAL, _pulse_cb);
            } else {
                Source.remove (_pulse_hid);
                _pulse_hid = 0;
                queue_draw ();
            }
        }
    }

    public override void dispose () {
        pulsing = false;
        base.dispose ();
    }

    public override void size_request (out Gtk.Requisition requisition) {
        base.size_request (out requisition);

        if (width > 0)
            requisition.width = width;
        if (height > 0)
            requisition.height = height;
    }

    public override bool expose_event (Gdk.EventExpose event) {
        base.expose_event (event);

        var attr = _attr;
        var cache = true;

        if (attr.width <= 0 || attr.height <= 0) {
            attr.width = allocation.width;
            attr.height = allocation.height;
            cache = false;
        }

        Cairo.Surface? surface = icon_get_surface (attr, cache);
        if (surface == null)
            return false;

        var xalign_ = (get_direction () != Gtk.TextDirection.LTR)
                ? xalign : 1.0f - xalign;

        var x = (int) Math.floor (allocation.x + xpad +
                (allocation.width - attr.width) * xalign_);
        var y = (int) Math.floor (allocation.y + ypad +
                (allocation.height - attr.height) * yalign);

        Cairo.Context cairo_context = Gdk.cairo_create (window);
        cairo_context.set_source_surface (surface, x, y);

        if (pulsing) {
            double alpha = 0.2 + (Math.sin (_pulse_phase) + 1.0) / 2.5;
            cairo_context.paint_with_alpha (alpha);
        } else if (get_state () == Gtk.StateType.INSENSITIVE)
            cairo_context.paint_with_alpha (0.25);
        else
            cairo_context.paint ();

        return false;
    }

    private bool _pulse_cb () {
        _pulse_phase += _PULSE_STEP;
        queue_draw ();
        return true;
    }

    private const int _PULSE_INTERVAL = 100;
    private const double _PULSE_STEP = Math.PI / 10.0;

    private IconAttr _attr;
    private string? _file;
    private uint _pulse_hid;
    private double _pulse_phase;
}

namespace Sugar {
    public struct IconAttr {
        /** Path to svg file to load */
        public string? file_name;
        /** If not default, change svg fill color */
        public Color fill_color;
        /** If not default, change svg stroke color */
        public Color stroke_color;
        /** If more then 0, change image width */
        public int width;
        /** If more then 0, change image height */
        public int height;

        public XoColor xo_color {
            set {
                fill_color = value.fill;
                stroke_color = value.stroke;
            }
        }

        /** Name of icon to load */
        public string icon_name {
            set {
                var theme = Gtk.IconTheme.get_default ();
                var info = theme.lookup_icon (value, width, 0);

                if (info != null)
                    file_name = info.get_filename ();
                else {
                    warning ("No icon with the name '%s' was found", value);
                    file_name = null;
                }
            }
        }

        /** Specify image width and height from icon size */
        public Gtk.IconSize icon_size {
            set {
                Gtk.icon_size_lookup (value, out width, out height);
            }
        }

        public static uint hash (IconAttr x) {
            return (x.file_name == null ? 0 : x.file_name.hash ()) +
                    x.fill_color.integer + x.stroke_color.integer +
                    x.width + x.height;
        }

        public static bool cmp (IconAttr x, IconAttr y) {
            return x.file_name == y.file_name && x.fill_color == y.fill_color &&
                    x.stroke_color == y.stroke_color && x.width == y.width &&
                    x.height == y.height;
        }
    }

    public void icon_cache_reset () {
        _surface_cache = null;
    }

    /**
     * Get cairo surface for sugar icon.
     *
     * @param attr  icon attributes
     * @param cache cache generated surface
     *
     * @return surface or null
     */
    public Cairo.Surface? icon_get_surface (IconAttr attr, bool cache = true) {
        if (attr.file_name == null)
            return null;

        if (_surface_cache == null)
            _surface_cache = new LRU<IconAttr?, Cairo.Surface> (
                    50, (HashFunc) IconAttr.hash, (EqualFunc) IconAttr.cmp);

        Cairo.Surface? surface = _surface_cache.get (attr);
        if (surface != null)
            return surface;

        var handle = _icon_load_svg (attr);
        if (handle == null)
            return null;

        int dst_width = (attr.width > 0) ? attr.width : handle.width;
        int dst_height = (attr.height > 0) ? attr.height : handle.height;

        surface = new Cairo.ImageSurface (Cairo.Format.ARGB32,
                dst_width, dst_height);
        var context = new Cairo.Context (surface);

        if (dst_width != handle.width || dst_height != handle.height)
            context.scale ((float) dst_width / handle.width,
                    (float) dst_height / handle.height);

        handle.render_cairo (context);

        if (cache)
            _surface_cache.set (attr, surface);

        return surface;
    }

    public Gdk.Pixbuf? icon_get_pixbuf (IconAttr attr) {
        if (attr.file_name == null)
            return null;

        var handle = _icon_load_svg (attr);
        if (handle == null)
            return null;

        int dst_width = (attr.width > 0) ? attr.width : handle.width;
        int dst_height = (attr.height > 0) ? attr.height : handle.height;

        handle.set_size_callback ((width, height) => {
            width = dst_width;
            height = dst_height;
        });

        return handle.get_pixbuf ();
    }

    private Rsvg.Handle? _icon_load_svg (IconAttr attr) {
        MappedFile? src = null;
        StringBuilder? dst = null;

        try {
            src = new MappedFile (attr.file_name, false);
        } catch (FileError e) {
            warning ("Cannot load '%s' svg: %s", attr.file_name, e.message);
            return null;
        }

        var pos = src.get_contents ();
        var end = src.get_contents () + src.get_length ();

        string? fill = attr.fill_color.alpha > 0.0
                ? attr.fill_color.html : null;
        string? stroke = attr.stroke_color.alpha > 0.0
                ? attr.stroke_color.html : null;

        if (fill != null || stroke != null) {
            for (var i = pos; i < end; i++) {
                if (to_string (i).has_prefix ("<svg"))
                    break;

                unowned string color = null;

                if (fill != null && to_string (i).has_prefix (_FILL_TAG)) {
                    color = fill;
                    i += _FILL_TAG.length;
                } else if (stroke != null &&
                        to_string (i).has_prefix (_STROKE_TAG)) {
                    color = stroke;
                    i += _STROKE_TAG.length;
                }

                if (color != null) {
                    if (dst == null)
                        dst = new StringBuilder.sized (src.get_length () +
                                (fill == null ? 0 : fill.length) +
                                (stroke == null ? 0 : stroke.length));

                    dst.append_len (to_string (pos), (ssize_t) (i - pos));
                    dst.append (color);

                    var end_i = to_string (i).index_of_char ('"');
                    if (end_i < 0) {
                        warning ("Wron svg file, cannot find second quotes");
                        return null;
                    }

                    i += end_i;
                    pos = i;
                }
            }
        }

        unowned uchar[] data;
        size_t data_len;

        if (pos == src.get_contents ()) {
            data = to_uchars (src.get_contents ());
            data_len = src.get_length ();
        } else {
            dst.append (to_string (pos));
            data = (uchar[]) dst.str;
            data_len = dst.len;
        }

        try {
            return new Rsvg.Handle.from_data (data, data_len);
        } catch (Error e) {
            warning ("Cannot parse '%s' svg: %s", attr.file_name, e.message);
            return null;
        }
    }

    private const string _FILL_TAG = "<!ENTITY fill_color \"";
    private const string _STROKE_TAG = "<!ENTITY stroke_color \"";

    private static LRU<IconAttr?, Cairo.Surface> _surface_cache;
}
