/* table.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Analog of Gtk.Table with auto column creation
 *
 * Before using Table widget, set width property.
 */
public class Sugar.Table : Sugar.Box {
    public int width {
        get {
            return _width;
        }
        set {
            if (value == width)
                return;
            _width = value;
            _queue_calc_layout ();
        }
    }

    construct {
        set_flags (Gtk.WidgetFlags.NO_WINDOW);
    }

    public override void add (Gtk.Widget widget) {
        base.add (widget);
        _queue_calc_layout ();
    }

    public override void remove (Gtk.Widget widget) {
        base.remove (widget);
        _queue_calc_layout ();
    }

    public override void size_request (out Gtk.Requisition requisition) {
        requisition.width = 0;
        requisition.height = 0;
        child_size_request (ref requisition);

        if (_postponed_layout)
            _calc_layout ();

        if (_postponed_layout) {
            requisition.width = 0;
            requisition.height = _cell_height * children_count;
        } else {
            int column_count = int.min (children_count,
                    int.max (1, _table_width / _cell_width));
            int row_count = children_count / column_count +
                    (children_count % column_count > 0 ? 1 : 0);
            requisition.width = width;
            requisition.height = _cell_height * row_count;
        }
    }

    public override void size_allocate (Gdk.Rectangle allocation) {
        this.allocation = (Gtk.Allocation) allocation;

        if (_postponed_layout) {
            var y = child_y;
            foreach (var child in children) {
                Gdk.Rectangle child_allocation = {
                    child_x, y, child_width, _cell_height
                };
                child.size_allocate (child_allocation);
                y += _cell_height;
            }
        } else {
            int column_count = int.min (children_count,
                    int.max (1, child_width / _cell_width));
            int column_width = child_width / column_count;

            unowned List<Gtk.Widget> child = children.first ();
            for (int y = 0; child != null; ++y)
                for (int x = 0; x < column_count; ++x)
                    if (child != null) {
                        Gdk.Rectangle child_allocation = {
                            child_x + x * column_width,
                            child_y + y * _cell_height,
                            column_width, _cell_height
                        };
                        child.data.size_allocate (child_allocation);
                        child = child.next;
                    }
        }
    }

    private int _table_width {
        get {
            return width - border_left - border_right - 2 * (int) border_width;
        }
    }

    private void _queue_calc_layout () {
        _postponed_layout = true;
        queue_resize ();
    }

    private void _calc_layout () {
        _cell_width = 0;
        _cell_height = 0;

        if (width <= 0) {
            warning ("Call set_size_request to setup width at first");
            return;
        }

        if (children_count == 0)
            return;

        var widths = new Array<int>.sized (false, false, (uint) sizeof (int),
                children_count);

        foreach (var child in children) {
            Gtk.Requisition child_requisition;
            child.size_request (out child_requisition);
            _cell_height = int.max (_cell_height, child_requisition.height);
            widths.append_val (child_requisition.width);
        }

        widths.sort ((a, b) => {
            return (int) a - (int) b;
        });

        // TODO count on width while choosing _cell_width
        int i = (int) (children_count / 4.0 * 3.0) - 1;
        _cell_width = widths.index (int.max (0, i));

        if (_cell_width > 0)
            _postponed_layout = false;
    }

    private int _cell_width = 0;
    private int _cell_height = 0;
    private int _width = -1;
    private bool _postponed_layout = true;
}
