#!/usr/bin/env python

import __init__

import os
import shutil
import gtk
import gobject
import unittest

import env, ds

class JobjectTest(unittest.TestCase):

    def setUp(self):
        self.trash = []
        shutil.rmtree('tmp', ignore_errors=True)
        os.makedirs('tmp')

    def tearDown(self):
        for i in self.trash:
            i.unlink()
        del self.trash

    def testCreate(self):
        jobject = ds.jobject_create()
        self.trash.append(jobject)

        self.assertEqual(None, jobject.props.uid)
        self.assertEqual(0, jobject.props.timestamp)
        self.assertEqual('Unknown Activity Activity', jobject.props.title)
        self.assertEqual(False, jobject.props.title_set_by_user)
        self.assertEqual('unknown.activity', jobject.props.activity)
        #self.assertEqual('testCreate', jobject.props.activity_id)
        self.assertEqual(False, jobject.props.keep)
        #self.assertEqual('', jobject.props.preview)
        self.assertEqual(env.SHARE_SCOPE_PRIVATE, jobject.props.share_scope)
        self.assertEqual(env.profile_get_color().to_string(),
                jobject.props.icon_color.to_string())

    def testWrite(self):
        jobject = ds.jobject_create()
        self.trash.append(jobject)

        errors = []

        def cb(__, error):
            errors.append(error)
            gtk.main_quit()
        jobject.connect('write-end', cb)

        jobject.write()
        gtk.main ()

        self.assertEqual([None], errors)
        self.assertNotEqual(None, jobject.props.uid)
        self.assertNotEqual(0, jobject.props.timestamp)

    def testFind(self):
        jobject_1 = ds.jobject_create()
        self.trash.append(jobject_1)

        def cb(__, error):
            gtk.main_quit()
        jobject_1.connect('write-end', cb)

        jobject_1.write()
        gtk.main ()

        jobject_2 = ds.jobject_find(jobject_1.props.uid)

        self.assertEqual(jobject_1.props.uid, jobject_2.props.uid)
        self.assertEqual(jobject_1.props.timestamp, jobject_2.props.timestamp)
        self.assertEqual(jobject_1.props.title, jobject_2.props.title)
        self.assertEqual(jobject_1.props.title_set_by_user,
                jobject_2.props.title_set_by_user)
        self.assertEqual(jobject_1.props.activity, jobject_2.props.activity)
        self.assertEqual(jobject_1.props.activity_id,
                jobject_2.props.activity_id)
        self.assertEqual(jobject_1.props.keep, jobject_2.props.keep)
        #self.assertEqual(jobject_1.props.preview, jobject_2.props.preview)
        self.assertEqual(jobject_1.props.share_scope,
                jobject_2.props.share_scope)
        self.assertEqual(jobject_1.props.icon_color.to_string(),
                jobject_2.props.icon_color.to_string())

    def testUpdate(self):
        jobject_1 = ds.jobject_create()
        self.trash.append(jobject_1)

        titles_1 = []
        titles_2 = []

        def write_end_cb(__, error):
            gtk.main_quit()
        jobject_1.connect('write-end', write_end_cb)

        def notify_title_1(__, error):
            titles_1.append(jobject_1.props.title)
        jobject_1.connect('notify::title', notify_title_1)

        jobject_1.write()
        gtk.main ()

        jobject_2 = ds.jobject_find(jobject_1.props.uid)
        titles = []

        jobject_2.connect('write-end', write_end_cb)
        def notify_title_2(__, error):
            titles_2.append(jobject_2.props.title)
        jobject_2.connect('notify::title', notify_title_2)

        jobject_1.props.title = 'title_1'
        jobject_1.write()
        gtk.main ()

        self.assertEqual(['title_1', 'title_1'], titles_1)
        self.assertEqual(['title_1'], titles_2)
        self.assertEqual('title_1', jobject_1.props.title)
        self.assertEqual('title_1', jobject_2.props.title)

        del titles_1[:]
        del titles_2[:]

        jobject_1.props.title = 'title_2'
        jobject_2.props.title = 'title_3'
        jobject_1.write()
        gtk.main ()

        self.assertEqual(['title_2', 'title_2'], titles_1)
        self.assertEqual(['title_3'], titles_2)
        self.assertEqual('title_2', jobject_1.props.title)
        self.assertEqual('title_3', jobject_2.props.title)

        del titles_1[:]
        del titles_2[:]

        jobject_2.write()
        gtk.main ()

        self.assertEqual(['title_3'], titles_1)
        self.assertEqual(['title_3'], titles_2)
        self.assertEqual('title_3', jobject_1.props.title)
        self.assertEqual('title_3', jobject_2.props.title)

        del titles_1[:]
        del titles_2[:]

        jobject_1.props.title = 'title_4'
        jobject_1.write()
        gtk.main ()

        self.assertEqual(['title_4', 'title_4'], titles_1)
        self.assertEqual(['title_4'], titles_2)
        self.assertEqual('title_4', jobject_1.props.title)
        self.assertEqual('title_4', jobject_2.props.title)

    def testWriteFile(self):
        jobject_1 = ds.jobject_create()
        self.trash.append(jobject_1)

        errors = []

        def cb(__, error):
            errors.append(error)
            gtk.main_quit()
        jobject_1.connect('write-end', cb)

        file_src = 'tmp/testWriteFile'
        file_1_handler = file(file_src, 'w')
        file_1_handler.write('testWriteFile')
        file_1_handler.close()

        jobject_1.write_file(file_src, False)
        gtk.main ()

        file_1 = jobject_1.props.file_path

        self.assertEqual([None], errors)
        self.assertEqual(True, os.path.exists(file_src))
        self.assertEqual(True, os.path.exists(file_1))
        self.assertNotEqual(file_src, file_1)
        self.assertEqual('testWriteFile', file(file_1).read())

        jobject_2 = ds.jobject_find(jobject_1.props.uid)
        file_2 = jobject_2.props.file_path

        self.assertEqual(True, os.path.exists(file_2))
        self.assertNotEqual(file_src, file_2)
        self.assertNotEqual(file_1, file_2)

        jobject_2 = None
        import gc
        gc.collect()

        self.assertEqual(False, os.path.exists(file_2))

    def testWriteFileOwnership(self):
        jobject = ds.jobject_create()
        self.trash.append(jobject)

        errors = []

        def cb(__, error):
            errors.append(error)
            gtk.main_quit()
        jobject.connect('write-end', cb)

        file_src = 'tmp/testWriteFileOwnership'
        file_handler = file(file_src, 'w')
        file_handler.write('testWriteFileOwnership')
        file_handler.close()

        jobject.write_file(file_src, True)
        gtk.main ()

        ds_file = jobject.props.file_path

        self.assertEqual([None], errors)
        self.assertEqual(False, os.path.exists(file_src))
        self.assertEqual(True, os.path.exists(ds_file))
        self.assertNotEqual(file_src, ds_file)
        self.assertEqual('testWriteFileOwnership', file(ds_file).read())

    def testWriteFileQueue(self):
        jobject = ds.jobject_create()
        self.trash.append(jobject)

        errors = []

        def cb(__, error):
            errors.append(error)
            gtk.main_quit()
        jobject.connect('write-end', cb)

        def touch(name):
            path = os.path.join('tmp', name)
            f = file(path, 'w')
            f.write(name)
            f.close
            return path

        file_1 = touch('1')
        file_2 = touch('2')

        jobject.write_file(file_1, False)
        jobject.write_file(file_2, False)

        gtk.main ()

        ds_file = jobject.props.file_path

        self.assertEqual([None], errors)
        self.assertEqual(True, os.path.exists(file_1))
        self.assertEqual(True, os.path.exists(file_2))
        self.assertEqual('2', file(ds_file).read())

        file_3 = touch('3')
        file_4 = touch('4')

        jobject.write_file(file_3, True)
        jobject.write_file(file_4, True)

        gtk.main ()

        self.assertEqual(False, os.path.exists(ds_file))
        ds_file = jobject.props.file_path

        self.assertEqual([None, None], errors)
        self.assertEqual(False, os.path.exists(file_3))
        self.assertEqual(False, os.path.exists(file_4))
        self.assertEqual('4', file(ds_file).read())

        file_5 = touch('5')
        file_6 = touch('6')
        file_7 = touch('7')

        jobject.write_file(file_5, False)
        jobject.write_file(file_6, True)
        jobject.write_file(file_7, False)

        gtk.main ()

        self.assertEqual(False, os.path.exists(ds_file))
        ds_file = jobject.props.file_path

        self.assertEqual([None, None, None], errors)
        self.assertEqual(True, os.path.exists(file_5))
        self.assertEqual(False, os.path.exists(file_6))
        self.assertEqual(True, os.path.exists(file_7))
        self.assertEqual('7', file(ds_file).read())

        file_8 = touch('8')

        jobject.write_file(file_8, False)
        jobject.write()

        gtk.main ()

        self.assertEqual(False, os.path.exists(ds_file))
        ds_file = jobject.props.file_path

        self.assertEqual([None, None, None, None], errors)
        self.assertEqual(True, os.path.exists(file_8))
        self.assertEqual('8', file(ds_file).read())

        file_9 = touch('9')

        jobject.write_file(file_9, True)
        jobject.write()

        gtk.main ()

        self.assertEqual(False, os.path.exists(ds_file))
        ds_file = jobject.props.file_path

        self.assertEqual([None, None, None, None, None], errors)
        self.assertEqual(False, os.path.exists(file_9))
        self.assertEqual('9', file(ds_file).read())

    def testCOW(self):
        jobject = ds.jobject_create()

        errors = []

        def cb(__, error):
            errors.append(error)
            gtk.main_quit()
        jobject.connect('write-end', cb)

        jobject.cow()
        gtk.main ()

        self.assertEqual([None], errors)
        self.assertEqual(None, jobject.props.uid)
        self.assertNotEqual(0, jobject.props.timestamp)

        jobject.write()
        gtk.main ()

        self.assertEqual([None, None], errors)
        self.assertNotEqual(None, jobject.props.uid)
        self.assertNotEqual(0, jobject.props.timestamp)

        jobject.cow()
        gtk.main ()

        self.assertEqual([None, None, None], errors)
        self.assertEqual(None, jobject.props.uid)
        self.assertNotEqual(0, jobject.props.timestamp)


if __name__ == '__main__':
    unittest.main()
