/* connection.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;

/**
 * Handle all things needed to make sugar shared connection
 *
 * The purpose of this class is to simplify Telepathy usage taking into
 * account activity development process i.e. targeting workflow is only
 * programmable Activity<->Activity interaction. It is not intended to support
 * all Telepathy features, if you need some of them, just fallback to pure
 * Telepathy.
 *
 * Class represents Telepathy connection (what ever it could mean). Only one
 * connection is allowed within activity launch session. Connection is
 * identified by Environ.activity_id value.
 *
 * To start interaction process between joined buddies, call on of offer_*
 * methods. Connection class supports two kinds of interactions, channels and
 * streams. Sugar.Channel class which is based of Telepathy DBus Tubes
 * (http://telepathy.freedesktop.org/spec/Channel_Type_DBus_Tube.html) and
 * intended to be programmable DBus based interfaces. Sugar.Stream class
 * (http://telepathy.freedesktop.org/spec/Channel_Type_Stream_Tube.html)
 * is for low level data transfer similar to sockets.
 */
public class Sugar.Connection : Object {
    /**
     * Connection scope was changed
     *
     * Signal will be sent not right after changing scope property but
     * when connection was really changed i.e. scope_changing property is false.
     *
     * @param prev_scope    Previous scope property value
     * @param error         If changing was a reaction on failure
     *                      localized text that could be used in GUI alerts
     */
    public signal void scope_changed (ShareScope prev_scope, string? error);

    /**
     * New channel is arrived
     *
     * @param channel   channel which just appeared
     */
    public signal void channel_appeared (Channel channel);

    /**
     * One of recent offer_channel calls is failed
     *
     * @param bus_name  argument passed to failed offer_channel invocation
     */
    public signal void offer_failed (string bus_name);

    /**
     * Channel was closed
     *
     * @param channel   closed channel, object is till valid but offline
     */
    public signal void channel_disappeared (Channel channel);

    /**
     * Current connection scope that was set by user
     *
     * After setting this property, Connection object will initiate scope
     * changing (scope_changing property will be set to true).  Thus, do not
     * judge about real connection scope on this value, listen this property
     * notification only for scope indication puposes. To do real connection
     * scope based work, connect to scope-changed signal.
     */
    public ShareScope scope {
        get {
            return _scope;
        }
        set {
            if (value == scope)
                return;

            if (scope_changing) {
                warning ("Cannot set new scope in scope_changing mode");
                return;
            }

            if (_connect ()) {
                debug (@"Setting share scope to $(_scope_names[value])");

                _scope = value;
                scope_changing = true;
                _backend.set_scope (scope);
            }
        }
    }

    /**
     * Connection is in scope changing mode
     *
     * This prerty is tru when scope value was changed and Connection object
     * is in low level scope changing mode. When low level scope is changed,
     * scope_changing signal will be sent and this property will be false.
     */
    public bool scope_changing {
        get { return _scope_changing; }
        private set { _scope_changing = value; }
    }

    /**
     * If current buddy is an initiator of shared connection
     *
     * If current user was the first who made shared connection online, this
     * property will be true. If while going online, this Connection object
     * joined to already existed public connection, this property will be false.
     */
    public bool initiator {
        get { return _backend.initiator; }
    }

    /**
     * Name of connection to advertise in the public
     *
     * This is the same value like "title" field in Journal entry.
     * Before switching to online modes, set this property. In most cases
     * it could be Title of activity. Also set this property after Title
     * was changed.
     */
    public string title {
        get {
            return _backend.title;
        }
        set {
            _backend.title = value;
            if (scope != ShareScope.PRIVATE)
                _backend.on_title_changing ();
        }
    }

    /**
     * Is connection already shared
     *
     * Property will return true If someone already has shared the same
     * activity_id connection.
     */
    public bool shared {
        get {
            if (!_connect ())
                return false;
            return _backend.is_shared ();
        }
    }

    construct {
        assert (Environ.initialized);

        _backend = new Collab.PS ();
        _backend.fallback_cb = _scope_fallback_cb;
        _backend.scope_changed_cb = _scope_changed_cb;
        _backend.channel_status_cb = _channel_status_cb;
        _backend.offer_failed = _offer_failed_cb;
    }

    /**
     * Create a channel to start interaction process
     *
     * To get Channel ready to use object that represents newly created channel,
     * connect to channel_appeared signal before invoking this method. If
     * channel for given name was already created, method will do nothing i.e.
     * won't emit channel_appeared signal (it was already emited). So, to make
     * robust code, connect to channel_appeared signal before making Connection
     * scope public and assume that you can get channel you need before invoking
     * this method.
     *
     * @param bus_name  A string representing the channel that will be used over
     *                  the connection. It should be a well-known D-Bus service
     *                  name, of the form org.example.ServiceName
     */
    public void offer_channel (string bus_name) {
        if (scope == ShareScope.PRIVATE)
            warning ("Cannot get channel if scope is private");
        else if (scope_changing)
            warning ("Cannot get channel if scope is changing");
        else
            _backend.offer_channel (bus_name);
    }

    private bool _connect () {
        if (_connected)
            return true;
        try {
            _connected = _backend.connect ();
        } catch (Error error) {
            warning ("Cannot connect: %s", error.message);
        }
        return _connected;
    }

    private void _scope_fallback_cb (ShareScope new_scope) {
        var prev_scope = scope;
        _scope = new_scope;
        notify_property ("scope");

        if (scope_changing) {
            _prev_scope = scope;
            scope_changing = false;
        }

        if (prev_scope != new_scope) {
            var error = _("Fallback to %s share scope").printf (
                    _(_scope_names[new_scope]));
            scope_changed (prev_scope, error);
        }
    }

    private void _scope_changed_cb (bool success) {
        if (!scope_changing)
            return;

        if (success) {
            scope_changing = false;
            scope_changed (_prev_scope, null);
            _prev_scope = scope;
        } else {
            var prev_scope = scope;
            _scope = _prev_scope;
            notify_property ("scope");
            scope_changing = false;

            var error = _("Cannot change share scope to %s").printf (
                    _(_scope_names[scope]));
            scope_changed (prev_scope, error);
        }

        debug (@"Share scope was set to $(_scope_names[scope])");
    }

    private void _channel_status_cb (Channel channel, bool online) {
        if (online) {
            debug ("Channel %s appeared", channel.bus_name);
            channel_appeared (channel);
        } else {
            debug ("Channel %s disappeared", channel.bus_name);
            channel_disappeared (channel);
        }
    }

    private void _offer_failed_cb (string bus_name) {
        debug ("Channel %s was failed to appear", bus_name);
        offer_failed (bus_name);
    }

    private static const string[] _scope_names = {
        N_("Private"),
        N_("Invite Only"),
        N_("Public")
    };

    private bool _connected;
    private ShareScope _scope = ShareScope.PRIVATE;
    private ShareScope _prev_scope = ShareScope.PRIVATE;
    private bool _scope_changing = false;
    private Collab.Backend _backend;
}
