/* style.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Port from original sugar-toolkit project.
 * File:   src/sugar/graphics/style.py
 * Commit: 6f3ecbdb1f765b4d1561525f7b868708488edde8
 *
 * Copyright (C) 2007, Red Hat, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

namespace Sugar {
    public struct Color {
        public const int BLACK = 0;
        public const int WHITE = 1;
        public const int TRANSPARENT = 2;
        public const int PANEL_GREY = 3;
        public const int SELECTION_GREY = 4;
        public const int TOOLBAR_GREY = 5;
        public const int BUTTON_GREY = 6;
        public const int INACTIVE_FILL = 7;
        public const int INACTIVE_STROKE = 8;
        public const int TEXT_FIELD_GREY = 9;
        public const int HIGHLIGHT = 10;
        public const int LAST = 11;

        public double alpha;
        public uint16 red;
        public uint16 green;
        public uint16 blue;

        public uint32 integer {
            get {
                return (uint32) (alpha * 255) +
                        (((uint32) blue & 0xFF) << 8) +
                        (((uint32) green & 0xFF) << 16) +
                        (((uint32) red & 0xFF) << 24);
            }
        }

        public string html {
            owned get {
                return "#%02x%02x%02x".printf (
                        red & 0xFF, green & 0xFF, blue & 0xFF);
            }
        }

        public string svg {
            owned get {
                if (alpha == 0.0)
                    return "none";
                else
                    return html;
            }
        }

        public static Color get (int type)
                requires (type >= 0 && type < LAST) {
            return _colors[type];
        }

        public string to_string () {
            return html;
        }

        /**
         * Parse color from string in #RRGGBB format
         */
        public static Color? from_string (string color_string) {
            var color = color_string.strip ();

            if (color[0] == '#')
                color = color.substring (1);

            if (color.length != 6) {
                warning (@"Color '$(color_string)' is not in #RRGGBB format");
                return null;
            }

            Color result = { };

            color.scanf ("%02hx%02hx%02hx",
                    &result.red, &result.green, &result.blue);

            /* Make it 1:1 to Gdk.Color */
            result.red |= result.red << 8;
            result.green |= result.green << 8;
            result.blue |= result.blue << 8;
            result.alpha = 1.0;

            return result;
        }

        private static const Color[] _colors = {
            /* BLACK */
            { 1.0, 0x0000, 0x0000, 0x0000 },
            /* WHITE */
            { 1.0, 0xFFFF, 0xFFFF, 0xFFFF },
            /* TRANSPARENT */
            { 0.0, 0xFFFF, 0xFFFF, 0xFFFF },
            /* PANEL_GREY */
            { 1.0, 0xC0C0, 0xC0C0, 0xC0C0 },
            /* SELECTION_GREY */
            { 1.0, 0xA6A6, 0xA6A6, 0xA6A6 },
            /* TOOLBAR_GREY */
            { 1.0, 0x2828, 0x2828, 0x2828 },
            /* BUTTON_GREY */
            { 1.0, 0x8080, 0x8080, 0x8080 },
            /* INACTIVE_FILL */
            { 1.0, 0x9D9D, 0x9F9F, 0xA1A1 },
            /* INACTIVE_STROKE */
            { 1.0, 0x7575, 0x7575, 0x7575 },
            /* TEXT_FIELD_GREY */
            { 1.0, 0xE5E5, 0xE5E5, 0xE5E5 },
            /* HIGHLIGHT */
            { 1.0, 0xE7E7, 0xE7E7, 0xE7E7 }
        };
    }

    public struct XoColor {
        public Color stroke;
        public Color fill;

        public string to_string () {
            return @"$(stroke),$(fill)";
        }

        public static XoColor? from_string (string color_string) {
            if (color_string == "white")
                return _white;
            else if (color_string == "insensitive")
                return _insensitive;

            var pair = color_string.split (",");
            if (pair.length != 2) {
                warning (@"Wrong XoColor pair '$(color_string)'");
                return null;
            }

            var stroke = Color.from_string (pair[0]);
            if (stroke == null) {
                warning (@"Fail to parse stroke color from '$(color_string)'");
                return null;
            }

            var fill = Color.from_string (pair[1]);
            if (fill == null) {
                warning (@"Fail to parse fill color from '$(color_string)'");
                return null;
            }

            XoColor result = { };
            result.stroke = stroke;
            result.fill = fill;

            return result;
        }

        private static const XoColor _white = {
            { 1.0, 0xFFFF, 0xFFFF, 0xFFFF },
            { 1.0, 0x4141, 0x4141, 0x4141 }
        };

        private static const XoColor _insensitive = {
            { 1.0, 0xFFFF, 0xFFFF, 0xFFFF },
            { 1.0, 0xE2E2, 0xE2E2, 0xE2E2 }
        };
    }

    public struct Font {
        public const int NORMAL = 0;
        public const int BOLD = 1;
        public const int LAST = 2;

        public string name {
            owned get {
                return _format.printf (face, size);
            }
        }

        public double size {
            get {
                try {
                    var conf = GConf.Client.get_default ();
                    return conf.get_float ("/desktop/sugar/font/default_size");
                } catch (Error e) {
                    debug ("Cannot get font size setting: %s.", e.message);
                    return 10;
                }
            }
        }

        public string face {
            owned get {
                try {
                    var conf = GConf.Client.get_default ();
                    return conf.get_string ("/desktop/sugar/font/default_face");
                } catch (Error e) {
                    debug ("Cannot get font face setting: %s.", e.message);
                    return "Bitstream Vera Sans";
                }
            }
        }

        public static Font get (int type)
                requires (type >= 0 && type < LAST) {
            return _fonts[type];
        }

        private static const Font[] _fonts = {
            /* NORMAL */
            { "%s %f" },
            /* BOLD */
            { "%s bold %f" }
        };

        private string _format;
    }

    public struct Metrics {
        [CCode (cname = "SUGAR_FOCUS_LINE_WIDTH")]
        public const int FOCUS_LINE_WIDTH = 0;
        [CCode (cname = "SUGAR_TOOLBAR_ARROW_SIZE")]
        public const int TOOLBAR_ARROW_SIZE = 1;
        [CCode (cname = "SUGAR_ZOOM_FACTOR")]
        public const int ZOOM_FACTOR = 2;
        [CCode (cname = "SUGAR_DEFAULT_SPACING")]
        public const int DEFAULT_SPACING = 3;
        [CCode (cname = "SUGAR_DEFAULT_PADDING")]
        public const int DEFAULT_PADDING = 4;
        [CCode (cname = "SUGAR_GRID_CELL_SIZE")]
        public const int GRID_CELL_SIZE = 5;
        [CCode (cname = "SUGAR_LINE_WIDTH")]
        public const int LINE_WIDTH = 6;
        [CCode (cname = "SUGAR_STANDARD_ICON_SIZE")]
        public const int STANDARD_ICON_SIZE = 7;
        [CCode (cname = "SUGAR_SMALL_ICON_SIZE")]
        public const int SMALL_ICON_SIZE = 8;
        [CCode (cname = "SUGAR_MEDIUM_ICON_SIZE")]
        public const int MEDIUM_ICON_SIZE = 9;
        [CCode (cname = "SUGAR_LARGE_ICON_SIZE")]
        public const int LARGE_ICON_SIZE = 10;
        [CCode (cname = "SUGAR_XLARGE_ICON_SIZE")]
        public const int XLARGE_ICON_SIZE = 11;
        [CCode (cname = "SUGAR_TOOLBOX_SEPARATOR_HEIGHT")]
        public const int TOOLBOX_SEPARATOR_HEIGHT = 12;
        [CCode (cname = "SUGAR_TOOLBOX_HORIZONTAL_PADDING")]
        public const int TOOLBOX_HORIZONTAL_PADDING = 13;
        [CCode (cname = "SUGAR_TOOLBOX_TAB_VBORDER")]
        public const int TOOLBOX_TAB_VBORDER = 14;
        [CCode (cname = "SUGAR_TOOLBOX_TAB_HBORDER")]
        public const int TOOLBOX_TAB_HBORDER = 15;
        [CCode (cname = "SUGAR_TOOLBOX_TAB_LABEL_WIDTH")]
        public const int TOOLBOX_TAB_LABEL_WIDTH = 16;
        [CCode (cname = "SUGAR_PALETTE_CURSOR_DISTANCE")]
        public const int PALETTE_CURSOR_DISTANCE = 17;
        [CCode (cname = "SUGAR_FONT_NORMAL_H")]
        public const int FONT_NORMAL_H = 18;
        [CCode (cname = "SUGAR_FONT_BOLD_H")]
        public const int FONT_BOLD_H = 19;
        public const int LAST = 20;

        public static double factor {
            get {
                return _get (ZOOM_FACTOR);
            }
        }

        public static int zoom (double units) {
            return (int) (_get (ZOOM_FACTOR) * units);
        }

        public static int get (int type)
                requires (type >= 0 && type < LAST) {
            return (int) _get (type);
        }

        private static double _get (int type) {
            if (_values[type].@value == 0.0) {
                switch (type) {
                case Metrics.ZOOM_FACTOR:
                    return _get_zoom_factor ();
                case Metrics.TOOLBOX_TAB_VBORDER:
                    return zoom (36) - _get (FONT_NORMAL_H) -
                            _get (FOCUS_LINE_WIDTH) / 2;
                case Metrics.TOOLBOX_TAB_HBORDER:
                    return zoom (15) - _get (FOCUS_LINE_WIDTH) -
                            1 /*_TAB_CURVATURE*/ ;
                default:
                    warning (@"Empty value for metrics type $(type)");
                    return 0;
                }
            } else if (_values[type].scaled)
                return zoom (_values[type].@value);
            else
                return _values[type].@value;
        }

        private static double _get_zoom_factor () {
            if (_zoom_factor > 0.0)
                return _zoom_factor;

            unowned string sugar_scale =
                    Environment.get_variable ("SUGAR_SCALING");

            if (sugar_scale == null)
                _zoom_factor = 1.0;
            else {
                var scale = int.parse (sugar_scale);
                if (scale > 0)
                    _zoom_factor = scale / 100.0;
                else {
                    warning (@"Invalid scaling SUGAR_SCALING=$sugar_scale.");
                    _zoom_factor = 1.0;
                }
            }

            return _zoom_factor;
        }

        private static double _zoom_factor = 0.0;

        private static const Metrics[] _values = {
            /* FOCUS_LINE_WIDTH */
            { false, 2.0 },
            /* TOOLBAR_ARROW_SIZE */
            { true, 24.0 },
            /* ZOOM_FACTOR */
            { },
            /* DEFAULT_SPACING */
            { true, 15.0 },
            /* DEFAULT_PADDING */
            { true, 6.0 },
            /* GRID_CELL_SIZE */
            { true, 75.0 },
            /* LINE_WIDTH */
            { true, 2.0 },
            /* STANDARD_ICON_SIZE */
            { true, 55.0 },
            /* SMALL_ICON_SIZE */
            { true, 55.0 * 0.5 },
            /* MEDIUM_ICON_SIZE */
            { true, 55.0 * 1.5 },
            /* LARGE_ICON_SIZE */
            { true, 55.0 * 2.0 },
            /* XLARGE_ICON_SIZE */
            { true, 55.0 * 2.75 },
            /* TOOLBOX_SEPARATOR_HEIGHT */
            { true, 9.0 },
            /* TOOLBOX_HORIZONTAL_PADDING */
            { true, 75.0 },
            /* TOOLBOX_TAB_VBORDER */
            { },
            /* TOOLBOX_TAB_HBORDER */
            { },
            /* TOOLBOX_TAB_LABEL_WIDTH */
            { true, 150.0 - 15.0 * 2.0 },
            /* PALETTE_CURSOR_DISTANCE */
            { true, 10.0 },
            /* FONT_NORMAL_H */
            { true, 24.0 },
            /* FONT_BOLD_H */
            { true, 24.0 }
        };

        private bool scaled;
        private double @value;
    }
}
