/* animator.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Port from original sugar-toolkit project.
 * File:   src/sugar/graphics/animator.py
 * Commit: c9e63eb8eadb0b133b88e9feb1ca48b75d959a7c
 *
 * Copyright (C) 2007, Red Hat, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * Port from original sugar-toolkit project.
 * File:   src/sugar/graphics/palettewindow.py
 * Commit: c684950ecff34e910e6da5840737a469f95a0e79
 *
 * Copyright (C) 2007, Eduardo Silva <edsiper@gmail.com>
 * Copyright (C) 2008, One Laptop Per Child
 * Copyright (C) 2009, Tomeu Vizoso
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

using Gee;

public enum Sugar.AnimationType {
    OUT_EXPO,
    IN_EXPO
}

public class Sugar.Animator : Object {
    public signal void next_frame (double frame);
    public signal void completed (bool aborted);

    public bool active {
        get { return _timeout_sid != 0; }
    }

    /**
     * @param duration sec
     */
    public Animator (double duration, int fps = 20, double start = 0.0,
            double stop = 1.0, AnimationType type = AnimationType.OUT_EXPO)
            requires (duration > 0.0)
            requires (fps > 0)
            requires (start >= 0.0 && start <= 1.0)
            requires (stop >= start && stop <= 1.0) {
        _duration = duration;
        _timeout = (int) (1.0 / fps * 1000);
        _start = start;
        _stop = stop;
        _type = type;
    }

    public void start () {
        if (_timeout_sid == 0) {
            _start_time = Timestamp ();
            _timeout_sid = Timeout.add (_timeout, _next_frame_cb);
        }
    }

    public void abort () {
        _complete (true);
    }

    private void _complete (bool aborted) {
        if (_timeout_sid == 0)
            return;
        Source.remove (_timeout_sid);
        _timeout_sid = 0;
        completed (aborted);
    }

    private bool _next_frame_cb () {
        var offset = double.min (_duration, Timestamp ().diff (_start_time));
        var the_rest = _stop - _start;
        var frame = _stop;

        if (offset < _duration) {
            if (_type == AnimationType.OUT_EXPO) {
                frame = the_rest *
                        (1 - Math.pow (2, -10 * offset / _duration)) + _start;
            } else if (_type == AnimationType.IN_EXPO) {
                frame = the_rest *
                        Math.pow (2, 10 * (offset / _duration - 1)) + _start;
            }
        }

        next_frame (frame);

        if (offset < _duration)
            return true;
        else {
            _complete (false);
            return false;
        }
    }

    private double _duration;
    private int _timeout;
    private double _start;
    private double _stop;
    private AnimationType _type;
    private uint _timeout_sid = 0;
    private Timestamp _start_time;
}

public class Sugar.MouseSpeedDetector : Object {
    public signal void motion_slow ();
    public signal void motion_fast ();

    /**
     * @param delay in msec
     * @param threshold in pixels (per tick of 'delay' msec)
     */
    public MouseSpeedDetector (int delay, int threshold) {
        _delay = delay;
        _threshold = threshold;
    }

    public void start () {
        stop ();
        _init_position ();
        _timeout_hid = Timeout.add (_delay, _timer_cb);
    }

    public void stop () {
        if (_timeout_hid != 0) {
            Source.remove (_timeout_hid);
            _timeout_hid = 0;
        }
        _state = 0;
    }

    private void _init_position () {
        var display = Gdk.Display.get_default ();
        display.get_pointer (null, out _x, out _y, null);
    }

    private bool _timer_cb () {
        var motion = _detect_motion ();

        if (motion && _state != _MOTION_FAST) {
            _state = _MOTION_FAST;
            motion_fast ();
        } else if (!motion && _state != _MOTION_SLOW) {
            _state = _MOTION_SLOW;
            motion_slow ();
        }

        return true;
    }

    private bool _detect_motion () {
        var x = _x;
        var y = _y;
        _init_position ();
        var dist2 = (x - _x) * (x - _x) + (y - _y) * (y - _y);
        return dist2 > _threshold * _threshold;
    }

    private const int _MOTION_SLOW = 1;
    private const int _MOTION_FAST = 2;

    private int _delay;
    private int _threshold;
    private int _state = 0;
    private uint _timeout_hid = 0;
    private int _x = 0;
    private int _y = 0;
}
