/* menu.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

public interface Sugar.MenuContainer : Object {
    public abstract Trigger menu_trigger { get; }
}

public class Sugar.MenuItem : Sugar.PaintBin {
    public const int SIZE = Metrics.STANDARD_ICON_SIZE;

    public Trigger? trigger { get; set; }

    public string? accelerator {
        get { return _accelerator.name; }
        set { _accelerator.name = value; }
    }

    public string label {
        get {
            return _label.label;
        }
        set {
            if (value == label)
                return;
            _box.remove (_label);
            _label_new (value);
        }
    }

    public int label_maxlen {
        set {
            _label.max_width_chars = value;
        }
    }

    public string? icon_name {
        get {
            return _icon.file;
        }
        set {
            if (value == icon_name)
                return;
            _icon.file = value;
            if (related_action is Gtk.RadioAction) {
                _icon_fill_color = _icon.fill_color;
                _icon_stroke_color = _icon.stroke_color;
            }
            _update_icon ();
        }
    }

    public Icon icon {
        get {
            return _icon;
        }
    }

    public Object tag { get; set; }

    public virtual signal void activate_item () {
        if (related_action != null)
            related_action.activate ();
        if (trigger != null)
            trigger.activate (this);
    }

    construct {
        radius = 0;
        padding_right = Metrics.get (Metrics.DEFAULT_PADDING) * 2;

        _accelerator = new Accelerator (this, "activate_item");

        _box = new Gtk.HBox (false, 0);
        _box.show ();
        add (_box);

        _icon = new Icon ();
        _icon.show ();

        _icon_box = new Gtk.Alignment (0.5f, 0.5f, 0, 0);
        _icon_box.set_size_request (Metrics.get (SIZE), Metrics.get (SIZE));
        _icon_box.show ();
        _box.pack_start (_icon_box, false, true, 0);

        _label_new ("");
        _selected = false;

        add_events (Gdk.EventMask.LEAVE_NOTIFY_MASK |
                Gdk.EventMask.ENTER_NOTIFY_MASK |
                Gdk.EventMask.BUTTON_PRESS_MASK |
                Gdk.EventMask.BUTTON_RELEASE_MASK);
    }

    public override void dispose () {
        related_action = null;
        base.dispose ();
    }

    public Gtk.Action related_action {
        get {
            return _related_action;
        }
        set {
            if (value == related_action)
                return;

            if (related_action != null) {
                related_action.notify["sensitive"].disconnect (
                        _related_action_sensitive_cb);
                related_action.activate.disconnect (
                        _related_action_activate_cb);
                related_action.disconnect_proxy (this);
            }

            _related_action = value;

            if (related_action != null) {
                label = related_action.label;
                icon_name = related_action.icon_name;
                if (icon_name == null && related_action is Gtk.ToggleAction)
                    icon_name = "emblem-favorite";
                _update_icon ();
                related_action.connect_proxy (this);
                related_action.notify["sensitive"].connect (
                        _related_action_sensitive_cb);
                related_action.activate.connect (_related_action_activate_cb);
            }
        }
    }

    /* Gtk.Widget */

    public override void hierarchy_changed (Gtk.Widget? previous_toplevel) {
        trigger = Trigger.find_dock (typeof (MenuContainer), parent,
                "menu_trigger");
    }

    public override bool button_release_event (Gdk.EventButton event) {
        if (event.button == 1) {
            activate_item ();
            return true;
        } else {
            return false;
        }
    }

    public override bool enter_notify_event (Gdk.EventCrossing event) {
        _selected = true;
        return false;
    }

    public override bool leave_notify_event (Gdk.EventCrossing event) {
        _selected = false;
        return false;
    }

    public override void map () {
        _selected = false;
        base.map ();
    }

    public bool _selected {
        set {
            var color = value ? Color.BUTTON_GREY : Color.BLACK;
            modify_bg (Gtk.StateType.NORMAL, color_type_to_rgb (color));
        }
    }

    private void _update_icon () {
        if (_icon.file == null) {
            if (_icon.parent != null)
                _icon_box.remove (_icon);
            return;
        }

        var toggle = related_action as Gtk.ToggleAction;

        if (toggle == null) {
            if (_icon.parent == null)
                _icon_box.add (_icon);
        } else {
            _icon.pixel_size = (int) (Metrics.get (SIZE) / 4.0 * 3.0);
            _icon.sensitive = toggle.sensitive;

            if (toggle is Gtk.RadioAction) {
                if (toggle.active) {
                    _icon.fill_color = _icon_fill_color;
                    _icon.stroke_color = _icon_stroke_color;
                } else {
                    _icon.stroke_color = Color.get (Color.BUTTON_GREY);
                    _icon.fill_color = Color.get (Color.TOOLBAR_GREY);
                }
                if (_icon.parent == null)
                    _icon_box.add (_icon);
            } else {
                if (toggle.active) {
                    if (_icon.parent == null)
                        _icon_box.add (_icon);
                } else {
                    if (_icon.parent != null)
                        _icon_box.remove (_icon);
                }
            }
        }
    }

    private void _label_new (string text) {
        _label = new Gtk.AccelLabel (text);
        _label.set_alignment (0.0f, 0.5f);
        _label.accel_widget = this;
        _label.set_ellipsize (Pango.EllipsizeMode.MIDDLE);
        _label.set_size_request (-1, Metrics.get (SIZE));
        label_maxlen = 60;
        _label.show ();
        _box.pack_end (_label, true, true, 0);
    }

    private void _related_action_activate_cb () {
        _update_icon ();
    }

    private void _related_action_sensitive_cb (Object sender, ParamSpec param) {
        _update_icon ();
    }

    private Gtk.Action _related_action;
    private Accelerator _accelerator;
    private Gtk.HBox _box;
    private Gtk.AccelLabel _label;
    private Icon _icon;
    private Color _icon_fill_color;
    private Color _icon_stroke_color;
    private Gtk.Alignment _icon_box;
}

public class Sugar.Menu : Sugar.Box {
    public int width {
        get {
            int width_request;
            get_size_request (out width_request, null);
            return width_request;
        }
        set {
            int height_request;
            get_size_request (null, out height_request);
            set_size_request (value, height_request);
        }
    }

    construct {
        set_flags (Gtk.WidgetFlags.NO_WINDOW);
    }

    /**
     * Place item to scrolled window
     *
     * @param widget        widget to insert to menu
     * @param max_row_count maximal height of scrolled window
     *                      in MenuItem's heights; <=0 to ignore
     * @param max_height    maximal height of scrolled window
     *                      <=0 to ignore
     * @param pos           position in menu to insert
     */
    public void insert_scrolled (Gtk.Widget widget, int max_row_count,
            int max_height, int pos = -1)
            requires (max_row_count > 0 || max_height > 0) {

        if (widget is Table && width >= 0)
            (widget as Table).width = width;

        int height = int.MAX, height_in_rows = int.MAX;

        if (max_row_count > 0)
            height_in_rows = max_row_count * Metrics.get (MenuItem.SIZE);
        if (max_height > 0) {
            max_height = int.max (Metrics.get (MenuItem.SIZE), max_height);
            height = max_height - max_height % Metrics.get (MenuItem.SIZE);
        }

        var scrolled = new _MenuScrolledItem (widget,
                int.min (height_in_rows, height));
        scrolled.show ();
        insert (scrolled, pos);
    }

    /**
     * Create and insert action based MenuItem
     *
     * @param action    action to make menu item based on
     * @param tag       tag for newly created menu item
     * @param pos       position in menu to insert
     */
    public void insert_action (Gtk.Action action, Object? tag = null,
            int pos = -1) {
        action.visible = true;
        var item = new MenuItem ();
        item.tag = tag;
        item.related_action = action;
        insert (item, pos);
    }

    public void insert_separator (int pos = -1) {
        insert (new _MenuSeparator (), pos);
    }

    public override void insert (Gtk.Widget widget, int pos) {
        if (widget is Table && width >= 0)
            (widget as Table).width = width;

        if (widget is _MenuScrolledItem || widget is _MenuSeparator)
            base.insert (widget, pos);
        else {
            var child = new Bin ();
            child.border = 0;
            child.border_left = Metrics.get (Metrics.DEFAULT_SPACING);
            child.border_right = Metrics.get (Metrics.DEFAULT_SPACING);
            child.show ();
            child.add (widget);
            base.insert (child, pos);
        }
    }

    public override void size_request (out Gtk.Requisition requisition) {
        requisition.width = 0;
        requisition.height = 0;

        foreach (var child in children) {
            Gtk.Requisition child_requisition;
            child.size_request (out child_requisition);
            requisition.width =
                    int.max (requisition.width, child_requisition.width);
            requisition.height += child_requisition.height;
        }

        child_size_request (ref requisition);
    }

    public override void size_allocate (Gdk.Rectangle allocation) {
        this.allocation = (Gtk.Allocation) allocation;

        var y = child_y;
        foreach (var child in children) {
            Gtk.Requisition child_requisition;
            child.get_child_requisition (out child_requisition);
            Gdk.Rectangle child_allocation = {
                child_x, y, child_width, child_requisition.height
            };
            child.size_allocate (child_allocation);
            y += child_requisition.height;
        }
    }
}

private class Sugar._MenuScrolledItem : Bin {
    public override int child_width {
        get {
            return base.child_width - _border_right;
        }
    }

    public _MenuScrolledItem (Gtk.Widget child, int max_height) {
        _child = child;
        _max_height = max_height;

        _scrolled = new Gtk.ScrolledWindow (null, null);
        _scrolled.hscrollbar_policy = Gtk.PolicyType.NEVER;
        _scrolled.vscrollbar_policy = Gtk.PolicyType.AUTOMATIC;
        _scrolled.add_with_viewport (_child);
        _scrolled.show ();
        add (_scrolled);

        border_left = Metrics.get (Metrics.DEFAULT_SPACING) -
                _scrolled.style.ythickness;
    }

    public override void size_request (out Gtk.Requisition requisition) {
        // Have to make fake call of ScrolledWindow.size_request
        // to not break its vscrollbar_policy logic
        Gtk.Requisition fake_requisition;
        _scrolled.size_request (out fake_requisition);

        _child.size_request (out requisition);
        if (requisition.height < _max_height)
            _border_right = border_left;
        else {
            requisition.height = int.min (_max_height, requisition.height);
            _border_right = 0;
        }
        requisition.height += 2 * _scrolled.style.ythickness;

        child_size_request (ref requisition);
    }

    private Gtk.Widget _child;
    private int _max_height;
    private Gtk.ScrolledWindow _scrolled;
    private int _border_right = 0;
}

private class Sugar._MenuSeparator : Gtk.SeparatorMenuItem {
}
