/* palette.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Port from original sugar-toolkit project.
 * File:   src/sugar/graphics/palette.py
 * Commit: 927c576e318a3aebd3217b35e1d4cf73582f0e4f
 *
 * Copyright (C) 2007, Eduardo Silva <edsiper@gmail.com>
 * Copyright (C) 2008, One Laptop Per Child
 * Copyright (C) 2009, Tomeu Vizoso
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

public class Sugar.Palette : Sugar.PaletteWindow, Sugar.MenuContainer {
    public signal void activate_menu (Gtk.Widget initiator);

    public Trigger menu_trigger {
        get { return _menu_trigger; }
    }

    public string? primary_text {
        get {
            return _primary_text;
        }
        set {
            _primary_text = value;

            if (value != null) {
                _label.set_markup (@"<b>$(value)</b>");
                _label.show ();
            }
        }
    }

    public string? secondary_text {
        get {
            return _secondary_text;
        }
        set {
            _secondary_text = value;

            if (secondary_text == null)
                _secondary_label.hide ();
            else {
                _secondary_label.set_text (secondary_text.delimit ("\n", '\0'));
                _secondary_label.show ();
            }
        }
    }

    public int text_maxlen {
        set {
            _label.max_width_chars = value;
            _secondary_label.max_width_chars = value;
        }
    }

    public new Icon? icon {
        get {
            return _icon;
        }
        set {
            if (icon != null)
                _icon_box.remove (icon);

            _icon = value;

            if (icon == null)
                _hide_icon ();
            else {
                icon.icon_size = Gtk.IconSize.LARGE_TOOLBAR;
                _icon_box.add (icon);
                icon.show ();
                _show_icon ();
            }
        }
    }

    public bool icon_visible {
        get {
            return _icon_visible;
        }
        set {
            _icon_visible = value;

            if (icon != null && icon_visible)
                _show_icon ();
            else
                _hide_icon ();
        }
    }

    public override Gtk.Widget? secondary {
        get {
            return _secondary.child;
        }
        set {
            if (value == secondary)
                return;

            if (secondary != null)
                _secondary.remove (secondary);

            if (value == null)
                secondary_visible = false;
            else {
                value.show ();
                _secondary.add (value);
            }

            _update_accept_focus ();
        }
    }

    public override bool secondary_visible {
        get { return _secondary_box.visible; }
        set { _secondary_box.visible = value; }
    }

    construct {
        var palette_box = new Gtk.VBox (false, 0);
        palette_box.show ();
        add (palette_box);

        _menu_trigger = new Trigger ();
        _menu_trigger.activate.connect (_menu_trigger_activate_cb);

        /* Primary box */

        var primary_box = new Gtk.HBox (false, 0);
        primary_box.set_size_request (-1, Metrics.get (Metrics.GRID_CELL_SIZE) -
                (int) border_width * 2);
        primary_box.show ();
        palette_box.pack_start (primary_box, false, true, 0);

        var icon_box = new Gtk.HBox (false, 0);
        icon_box.set_size_request (Metrics.get (Metrics.GRID_CELL_SIZE), -1);
        primary_box.pack_start (icon_box, false, true, 0);

        _icon_box = new Gtk.EventBox ();
        _icon_box.button_release_event.connect (_icon_button_release_event_cb);
        _icon_box.show ();
        icon_box.pack_start (_icon_box, true, true, 0);

        var labels_box = new Gtk.VBox (false, 0);
        labels_box.show ();
        _label_alignment = new Gtk.Alignment (0.0f, 0.5f, 1.0f, 0.33f);
        _label_alignment.set_padding (0, 0,
                Metrics.get (Metrics.DEFAULT_SPACING),
                Metrics.get (Metrics.DEFAULT_SPACING));
        _label_alignment.add (labels_box);
        _label_alignment.show ();
        primary_box.pack_start (_label_alignment, true, true, 0);

        _label = new Gtk.AccelLabel ("");
        _label.set_alignment (0.0f, 0.5f);
        _label.set_ellipsize (Pango.EllipsizeMode.MIDDLE);
        labels_box.pack_start (_label, true, true, 0);

        _secondary_label = new Gtk.Label (null);
        _secondary_label.set_alignment (0.0f, 0.5f);
        _secondary_label.set_ellipsize (Pango.EllipsizeMode.END);
        labels_box.pack_start (_secondary_label, true, true, 0);

        text_maxlen = 60;

        /* Secondary box */

        _secondary_box = new Gtk.VBox (false, 0);
        palette_box.pack_start (_secondary_box, true, true, 0);

        var separator = new Gtk.HSeparator ();
        separator.show ();
        _secondary_box.pack_start (separator, true, true, 0);

        _secondary = new Bin ();
        _secondary.border_top = Metrics.get (Metrics.DEFAULT_SPACING);
        _secondary.border_bottom = Metrics.get (Metrics.DEFAULT_SPACING);
        _secondary.show ();
        _secondary_box.pack_start (_secondary, true, true, 0);
    }

    public override void size_request (out Gtk.Requisition requisition) {
        base.size_request (out requisition);

        // gtk.AccelLabel request doesn't include the accelerator.
        Gtk.Requisition label_requisition;
        _label_alignment.size_request (out label_requisition);
        var label_width = label_requisition.width +
                (int) _label.get_accel_width () + (int) border_width * 2;

        requisition.width = int.max (requisition.width, label_width);
    }

    public override void realize () {
        base.realize ();
        _update_accept_focus ();
    }

    public override void style_set (Gtk.Style? previous_style) {
        base.style_set (previous_style);
        set_border_width (style.xthickness);
    }

    public override void show () {
        _label.accel_widget = connector.subject as Gtk.Widget;
        base.show ();
    }

    private void _show_icon () {
        _label_alignment.set_padding (0, 0, 0,
                Metrics.get (Metrics.DEFAULT_SPACING));
        _icon_box.parent.show ();
    }

    private void _hide_icon () {
        _icon_box.parent.hide ();
        _label_alignment.set_padding (0, 0,
                Metrics.get (Metrics.DEFAULT_SPACING),
                Metrics.get (Metrics.DEFAULT_SPACING));
    }

    private void _update_accept_focus () {
        if (is_realized ())
            window.set_accept_focus (secondary != null);
    }

    private bool _icon_button_release_event_cb (Gdk.EventButton event) {
        activate ();
        return false;
    }

    private void _menu_trigger_activate_cb (Gtk.Widget initiator) {
        hide ();
        activate_menu (initiator);
    }

    private string _primary_text;
    private string _secondary_text;
    private Icon _icon;
    private bool _icon_visible = true;
    private Gtk.EventBox _icon_box;
    private Gtk.Alignment _label_alignment;
    private Gtk.AccelLabel _label;
    private Gtk.Label _secondary_label;
    private Gtk.VBox _secondary_box;
    private Bin _secondary;
    private Trigger _menu_trigger;
}
