/* activity.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * High level class that accumulates non-GUI objects needed for regular activity
 *
 * Besides, Activity follows regular sugar workflows. It could be useful for
 * native sugar activities. In other cases, classes like Jobject, Shell etc.
 * could be used directly.
 */
public class Sugar.Activity : Object {
    /**
     * Journal object activity was launched to change
     *
     * Property could be null if Activity object was created without Journal
     * support i.e. use_journal constructor argument was false.
     */
    public Jobject? jobject { get; private set; }

    /**
     * Object to interact with Sugar Shell
     */
    public Shell shell { get; private set; }

    /**
     * Object to interact with Sugar Journal
     */
    public Journal journal { get; private set; }

    /**
     * Activity sharing session
     *
     * Property could be null if Activity object was created without sharing
     * support i.e. use_sharing constructor argument was false.
     */
    public Connection? connection { get; private set; }

    /**
     * Was activity resumed from Journal entry or not
     */
    public bool resumed {
        get { return _resumed; }
    }

    /**
     * Create activity instance
     *
     * @param use_journal   is activity going to (re)store Journal objects
     *                      if true, jobject will be set
     * @param use_sharing   is activity going to be shared
     *                      if true, connection will be set
     */
    public Activity (bool use_journal = true, bool use_sharing = true) {
        assert (Environ.initialized);

        shell = new Shell ();
        journal = new Journal ();

        if (use_journal) {
            if (Environ.object_id != null)
                jobject = Jobject.find (Environ.object_id);

            if (jobject != null) {
                debug ("Journal entry %s was found", jobject.uid);
                _resumed = true;
            } else {
                debug ("Create new Journal entry");
                jobject = Jobject.create ();
            }
        }

        if (use_sharing) {
            connection = new Connection ();

            if (jobject != null) {
                jobject.notify["title"].connect (() => {
                    connection.title = jobject.title;
                });
                connection.title = jobject.title;
                if (resumed)
                    connection.scope = jobject.share_scope;
                connection.scope_changed.connect (_scope_changed_cb);
            }

            if (!resumed) {
                if (!connection.shared)
                    debug ("Activity isn't shared, will start in private mode");
                else {
                    debug ("Activity is already shared, will join");
                    connection.scope = ShareScope.PUBLIC;
                }
            }
        }
    }

    public override void dispose () {
        if (jobject != null && !resumed && jobject.uid != null)
            journal.show_object (jobject.uid);
        base.dispose ();
    }

    /**
     * Write file to Journal object
     *
     * Makes sense only if Activity object was created with Journal support
     * i.e. use_journal constructor argument was true.
     *
     * @param file_path             path to the file to write
     * @param transfer_ownership    should sugar-datastore take ownership
     *                              of passed file_path
     */
    public virtual void write_file (string file_path,
            bool transfer_ownership = false)
            requires (jobject != null) {
        try {
            jobject.preview = get_preview ();
        } catch (Error error) {
            warning ("Cannot get preview: %s", error.message);
        }
        jobject.write_file (file_path, transfer_ownership);
    }

    void _scope_changed_cb (ShareScope prev_state, string? error) {
        jobject.share_scope = connection.scope;
    }

    private bool _resumed;
}
