/* ps.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;

/**
 * Sugar Presence Service backend
 *
 * Backend is tied to Telepathy API used in PS time.
 */
private class Sugar.Collab.PS : Sugar.Collab.Backend {
    public PS () {
        _tubes = new HashMap<uint, _DBusTube> ();
        _offers = new LinkedList<string> ();
        // For unit testing puposes, to have Connection instances with
        // different activity_id
        _activity_id = Environ.activity_id;
    }

    ~PS () {
        _abort ();
    }

    public override bool connect () throws DBus.Error {
        _bus = DBus.Bus.get (DBus.BusType.SESSION);

        // Monitor PS going online/offline
        _bus_object = _bus.get_object ("org.freedesktop.DBus",
                "/org/freedesktop/DBus", "org.freedesktop.DBus");
        _bus_object.NameOwnerChanged.connect (_NameOwnerChanged_cb);

        return true;
    }

    public override void set_scope (ShareScope scope) {
        switch (scope) {
        case ShareScope.PRIVATE:
            if (_activity_object != null)
                _leave (_OFFLINE_cb);
            else
                _OFFLINE_cb ();
            break;
        case ShareScope.INVITE_ONLY:
            if (_activity_object != null)
                _set_private (true);
            else
                _share (true);
            break;
        case ShareScope.PUBLIC:
            if (_activity_object != null)
                _set_private (false);
            else
                _join ();
            break;
        }
    }

    public override void on_title_changing () {
        var prop = Value (typeof (string));
        prop.set_string (title);
        _set_prop ("name", prop);
    }

    public override void offer_channel (string bus_name) {
        try {
            var props = new HashTable<string, Value?> (str_hash, str_equal);
            _tube_channel.OfferDBusTube (bus_name, props, _OfferDBusTube_cb);
            _offers.offer_head (bus_name);
        } catch (DBus.Error error) {
            warning ("Cannot offer dbus tube: %s", error.message);
            offer_failed (bus_name);
        }
    }

    private dynamic DBus.Object? _ps {
        get {
            if (_ps_object != null)
                return _ps_object;
            _ps_object = _bus.get_object (_NAME, _PATH, _INTERFACE);
            return _ps_object;
        }
        set {
            _ps_object = value;
        }
    }

    private bool _joined {
        get {
            if (_activity_object == null)
                return false;

            try {
                DBus.ObjectPath owner = _ps.GetOwner ();
                DBus.ObjectPath[] buddies =
                        _activity_object.GetJoinedBuddies ();
                foreach (var i in buddies)
                    if (i == owner)
                        return true;
                return false;
            } catch (DBus.Error error) {
                debug ("Cannot check if activity joined: %s", error.message);
                return false;
            }
        }
    }

    private void _share (bool is_private) {
        if (_ps == null) {
            scope_changed_cb (false);
            return;
        }

        var props = new HashTable<string, Value?> (str_hash, str_equal);
        var private_prop = Value (typeof (bool));
        private_prop.set_boolean (is_private);
        props.insert ("private", private_prop);

        try {
            _ps.ShareActivity (_activity_id, Environ.bundle_id, title, props,
                    _ShareActivity_cb);
        } catch (DBus.Error error) {
            warning ("Cannot share activity: %s", error.message);
            _abort ();
        }

        initiator = true;
    }

    public override bool is_shared () {
        try {
            DBus.ObjectPath activity_path;
            _ps.GetActivityById (_activity_id, out activity_path);
            return true;
        } catch (DBus.Error error) {
            return false;
        }
    }

    private void _join () {
        if (_ps == null) {
            scope_changed_cb (false);
            return;
        }

        DBus.ObjectPath activity_path;

        try {
            _ps.GetActivityById (_activity_id, out activity_path);
        } catch (DBus.Error error) {
            debug ("No shared activity, will create one: %s", error.message);
            _share (false);
            return;
        }

        debug ("Got existed activity path: %s", activity_path);
        _activity_object = _bus.get_object (_NAME, activity_path,
                _ACTIVITY_INTERFACE);

        if (_joined) {
            debug ("Activity is already joined");
            _setup_activity_object ();
        } else {
            try {
                _activity_object.Join (_Join_cb);
            } catch (DBus.Error error) {
                warning ("Cannot join to activity: %s", error.message);
                _abort ();
            }
        }

        initiator = false;
    }

    private void _set_prop (string prop, Value @value) {
        var props = new HashTable<string, Value?> (str_hash, str_equal);
        props.insert (prop, @value);

        try {
            _activity_object.SetProperties (props);
        } catch (DBus.Error error) {
            warning ("Cannot set activity %s property to %s",
                    prop, @value.strdup_contents ());
        }
    }

    private void _set_private (bool is_private) {
        var props = new HashTable<string, Value?> (str_hash, str_equal);
        var private_prop = Value (typeof (bool));
        private_prop.set_boolean (is_private);
        props.insert ("private", private_prop);

        try {
            _activity_object.SetProperties (props, _SetProperties_cb);
        } catch (DBus.Error error) {
            warning ("Cannot set activity private property to %d",
                    (int) is_private);
            _abort ();
            return;
        }
    }

    private void _leave (LeaveCallback leave_cb) {
        _leave_cb = leave_cb;

        try {
            _activity_object.Leave (_Leave_cb);
        } catch (DBus.Error error) {
            _Leave_cb (error);
        }
    }

    private void _abort () {
        debug ("Abort connection");

        if (_activity_object != null) {
            try {
                _activity_object.Leave ();
            } catch (DBus.Error error) {
                warning ("Cannot Leave activity on abort: %s", error.message);
            }
            _tube_channel = null;
            _activity_object = null;
        }

        fallback_cb (ShareScope.PRIVATE);
    }

    private void _setup_activity_object () {
        try {
            _activity_object.ListChannels (_ListChannels_cb);
        } catch (DBus.Error error) {
            warning ("Cannot list activity channels: %s", error.message);
            _abort ();
        }
    }

    private void _OfferDBusTube_cb (uint id, Error error) {
        var bus_name = _offers.poll_head ();

        if (error == null)
            debug ("Got %u offerd tube for %s", id, bus_name);
        else {
            warning ("Cannot offer dbus tube: %s", error.message);
            offer_failed (bus_name);
        }
    }

    private void _Join_cb (Error error) {
        if (error == null) {
            debug ("Joined to activity");
            _setup_activity_object ();
        } else {
            warning ("Cannot join to activity: %s", error.message);
            _abort ();
        }
    }

    private void _ListChannels_cb (string tp_conn_name,
            DBus.ObjectPath tp_conn_path, ValueArray[] channels, Error error) {
        if (error == null) {
            foreach (unowned ValueArray channel in channels) {
                var handle_type = channel.values[2].get_uint ();
                if (handle_type != Telepathy.HANDLE_TYPE_ROOM)
                    continue;

                DBus.ObjectPath * path = channel.values[0].get_boxed ();
                var iface = channel.values[1].get_string ();

                debug ("Got channel bus_name=%s path=%s iface=%s",
                        tp_conn_name, path, iface);

                if (iface == Telepathy.CHANNEL_TYPE_TUBES) {
                    _tube_channel = _bus.get_object (tp_conn_name, path, iface);
                    _tube_group = _bus.get_object (tp_conn_name, path,
                            Telepathy.CHANNEL_INTERFACE_GROUP);
                    try {
                        _tube_channel.ListTubes (_ListTubes_cb);
                    } catch (Error error) {
                        warning ("Cannot list tubes: %s", error.message);
                        _abort ();
                        return;
                    }
                }
            }
            if (_tube_channel == null) {
                warning ("Tubes channel was not found");
                _abort ();
                return;
            }
        } else {
            warning ("Cannot list activity channels: %s", error.message);
            _abort ();
            return;
        }
    }

    private void _ListTubes_cb (ValueArray[] tubes, Error error) {
        if (error == null) {
            _tube_channel.NewTube.connect (_NewTube_cb);
            _tube_channel.TubeStateChanged.connect (_TubeStateChanged_cb);
            _tube_channel.TubeClosed.connect (_TubeClosed_cb);

            foreach (unowned ValueArray tube in tubes) {
                var id = tube.values[0].get_uint ();
                var initiator = tube.values[1].get_uint ();
                var type = tube.values[2].get_uint ();
                var bus_name = tube.values[3].get_string ();
                var state = tube.values[5].get_uint ();

                debug ("Got existed tube id=%u initiator=%u type=%u " +
                        "bus_name=%s state=%u", id, initiator, type, bus_name,
                        state);

                if (type == Telepathy.TUBE_TYPE_DBUS) {
                    if (state == Telepathy.TUBE_STATE_LOCAL_PENDING)
                        _accept_tube (id);
                    else if (state == Telepathy.TUBE_STATE_OPEN)
                        _create_tube (id);
                }
            }

            scope_changed_cb (true);
        } else {
            warning ("Cannot get activity tubes list: %s", error.message);
            _abort ();
        }
    }

    private void _TubeClosed_cb (dynamic DBus.Object? channel, uint id) {
        if (!_tubes.has_key (id))
            warning ("Closed tube was not counted id=%u", id);
        else {
            channel_status_cb (_tubes[id], false);
            _tubes.unset (id);
            warning ("Tube closed id=%u", id);
        }
    }

    private void _TubeStateChanged_cb (dynamic DBus.Object? channel, uint id,
            uint state) {
        debug ("Tube status changed id=%u state=%u", id, state);
        if (state == Telepathy.TUBE_STATE_OPEN)
            _create_tube (id);
    }

    private void _accept_tube (uint id) {
        try {
            string bus_address;
            bus_address = _tube_channel.AcceptDBusTube (id);
        } catch (Error error) {
            warning ("Cannot accept tube: %s", error.message);
        }
    }

    private void _create_tube (uint id) {
        try {
            var tube = new _DBusTube (_tube_channel, _tube_group, id);
            _tubes[id] = tube;
            channel_status_cb (tube, true);
        } catch (Error error) {
            warning ("Cannot create tube: %s", error.message);
        }
    }

    private void _NewTube_cb (dynamic DBus.Object? channel, uint id,
            uint initiator, uint type, string bus_name,
            HashTable<string, Value? >? props, uint state) {
        debug ("Got new tube id=%u initiator=%u type=%u bus_name=%s state=%u",
                id, initiator, type, bus_name, state);

        if (type != Telepathy.TUBE_TYPE_DBUS)
            return;

        if (state == Telepathy.TUBE_STATE_LOCAL_PENDING)
            _accept_tube (id);
    }

    private void _SetProperties_cb (Error error) {
        if (error == null)
            scope_changed_cb (true);
        else {
            warning ("Fail to set activity private property: %s",
                    error.message);
            _abort ();
        }
    }

    private void _Leave_cb (Error error) {
        LeaveCallback leave_cb = _leave_cb;
        _leave_cb = null;

        if (error != null)
            warning ("Cannot leave activity, will abandon: %s", error.message);
        _tube_channel = null;
        _activity_object = null;

        if (leave_cb != null)
            leave_cb ();
    }

    private void _OFFLINE_cb () {
        _ps = null;
        scope_changed_cb (true);
    }

    private void _ShareActivity_cb (DBus.ObjectPath path, Error error) {
        if (error == null) {
            debug ("Got activity path: %s", path);
            _activity_object = _bus.get_object (_NAME, path,
                    _ACTIVITY_INTERFACE);
            _setup_activity_object ();
        } else {
            warning ("Fail to share activity: %s", error.message);
            _abort ();
        }
    }

    private void _NameOwnerChanged_cb (DBus.Object sender, string bus_name,
            string old_owner, string new_owner) {
        if (bus_name != _NAME)
            return;

        if (old_owner == "" && new_owner != "") {
            if (_ps != null)
                debug ("Detected PS going online");
        } else if (old_owner != "" && new_owner == "") {
            debug ("Detected PS going offline");
            _abort ();
        }
    }

    private delegate void LeaveCallback ();

    private static const string _NAME = "org.laptop.Sugar.Presence";
    private static const string _INTERFACE = "org.laptop.Sugar.Presence";
    private static const string _PATH = "/org/laptop/Sugar/Presence";
    private static const string _ACTIVITY_INTERFACE =
            "org.laptop.Sugar.Presence.Activity";

    private string _activity_id;
    private DBus.Connection _bus;
    private dynamic DBus.Object _bus_object;
    private dynamic DBus.Object _ps_object;
    private dynamic DBus.Object _activity_object;
    private dynamic DBus.Object _tube_channel;
    private dynamic DBus.Object _tube_group;
    private LeaveCallback _leave_cb = null;
    private HashMap<uint, _DBusTube> _tubes;
    private LinkedList<string> _offers;
}

private class Sugar.Collab._DBusTube : Sugar.Channel {
    public _DBusTube (dynamic DBus.Object channel, dynamic DBus.Object group,
            uint id) throws Error {
        _id = id;
        _channel = channel;
        _group = group;

        _self_handle = _group.GetSelfHandle ();
        address = _channel.GetDBusTubeAddress (_id);

        ValueArray[] tubes = _channel.ListTubes ();
        foreach (unowned ValueArray tube in tubes) {
            if (tube.values[0].get_uint () == _id) {
                _initiator_id = tube.values[1].get_uint ();
                bus_name = tube.values[3].get_string ();
                break;
            }
        }

        _channel.DBusNamesChanged.connect (_DBusNamesChanged_cb);
        _channel.GetDBusNames (_id, _GetDBusNames_cb);
    }

    private void _DBusNamesChanged_cb (dynamic DBus.Object channel,
            uint id, ValueArray[] added, uint[] removed) {
        debug ("Got buddies change tube_id=%u added.len=%d removed.len=%d",
                id, added.length, removed.length);

        if (id != _id)
            return;

        foreach (unowned ValueArray i in added) {
            var buddy_handle = i.values[0].get_uint ();
            var buddy_name = i.values[1].get_string ();
            _add_buddy (buddy_handle, buddy_name);
        }

        _remove_buddies (removed);
    }

    private void _GetDBusNames_cb (ValueArray[] buddies, Error error) {
        if (error == null) {
            uint[] removed = { };
            _DBusNamesChanged_cb (_channel, _id, buddies, removed);
        } else {
            warning ("Fail to call GetDBusNames: %s", error.message);
        }
    }

    private uint _id;
    private dynamic DBus.Object _channel;
    private dynamic DBus.Object _group;
}
