/* query.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

public errordomain Sugar.DataError {
}

/**
 * Base class for different kinds of possible data sources
 */
public class Sugar.DataQuery : Object {
    /**
     * Content of DataSource was changed
     *
     * This is a common signal which will be sent after setting query property
     * or after that something was changed in data e.g. added|removed|changed
     * data entries. The regular reaction on this signal is re-reading
     * DataSource entirely content.
     */
    public signal void changed ();

    /**
     * Next porsion of data entries was fetched
     *
     * @param success   entries were successfully fetched
     *
     * If get_async method was used, this signal will be emited after fetching
     * entries from the server.
     */
    public signal void got_entries (bool success);

    /**
     * Number of entries that are covered by the DataSource
     */
    public uint length {
        get { return _resultset == null ? 0 : _resultset.length; }
    }

    /**
     * Field name to order entries in the DataSource
     *
     * Field name could be prefixed by '+' or '-' symbol to change sorting
     * direction. Note that not all sugar-datastore support orderring.
     */
    public string order_by {
        get {
            return _order_by;
        }
        set {
            if (value != order_by) {
                _order_by = value;
                _do_query ();
            }
        }
    }

    /**
     * What kind of entries could be gotten from the DataSource
     *
     * On every query's properties change, DataSource will ask low level
     * implementation about new number of entries then will emit changed signal.
     */
    public DataQuery query {
        get { return _query; }
    }

    /**
     * Make sync call to get entry from the DataSource
     *
     * @param offset    position of entry to get starting from 0
     * @return          Jobject that represents entry, throw DataError otherwise
     *
     * Method can return entry immediately if it is cached or will wait
     * until low level implementation returns entry from the server.
     */
    public Jobject get (uint offset) throws (DataError) {
    }

    /**
     * Make async call to get entry from the DataSource
     *
     * @param offset    position of entry to get starting from 0
     * @return          Jobject that represents entry, null otherwise
     *
     * Method can return entry immediately if it is cached or will return
     * null and will ask level implementation to get entry from the server,
     * then will emit got_entries signal.
     */
    public Jobject? get_async (uint offset) {
    }

    private string _order_by = "-mtime";
}

public class Sugar.DataQuery : Object {
  def set_query (self, query):
    if (query == self._query)
      :
      return self._query = query if 'query' in query and not query['query'].startswith ('"'):
          query_text = '' for word in query['query'].split (' '):
              if not word:
                  continue if query_text:
                        query_text += ' '
                                query_text += word + '*'
                                self._query['query'] = query_text
                                self._do_query ()
                         query = property (get_query, set_query)}
