/* misc.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;

/**
 * State of activity sharing
 */
public enum Sugar.ShareScope {
    /** No connection */
    PRIVATE,
    /** Not visible shared connection, only invites */
    INVITE_ONLY,
    /** Fully visible shared connection */
    PUBLIC,
    /** Stopper value */
    LAST
}

/**
 * Least Recently Used queue
 *
 * Simple implementation of a length-limited O (1) LRU queue.
 * Adapted version from PyPE, http://pype.sourceforge.net.
 */
public class Sugar.LRU<K, V> {
    public LRU (int capacity, HashFunc? key_hash_func = null,
            EqualFunc? key_equal_func = null) {
        _queue = new HashMap<K, _Node<K, V>> (key_hash_func, key_equal_func);
        this.capacity = int.max (1, capacity);
    }

    public int capacity { get; private set; }

    public bool has_key (K key) {
        return _queue.has_key (key);
    }

    public V @get (K key) {
        var node = _queue.@get (key);
        if (node == null) {
            return null;
        } else {
            _actualize_node (node);
            return node.value;
        }
    }

    public void @set (K key, V value) {
        remove (key);

        var node = new _Node<K, V> ();
        node.key = key;
        node.value = value;

        _actualize_node (node);
        _queue[key] = node;

        if (_queue.size > capacity)
            _remove_node (_first);
    }

    public void remove (K key) {
        var node = _queue.@get (key);
        if (node == null)
            return;
        _remove_node (node);
    }

    private void _actualize_node (_Node<K, V> node) {
        if (_first == null)
            _first = node;
        if (_last != null)
            _last.next = node;
        node.prev = _last;
        node.next = null;
        _last = node;
    }

    private void _remove_node (_Node<K, V> node) {
        if (node.prev != null)
            node.prev.next = node.next;
        else
            _first = node.next;
        if (node.next != null)
            node.next.prev = node.prev;
        else
            _last = node.prev;
        _queue.unset (node.key);
    }

    class _Node<K, V> {
        public _Node<K, V> prev;
        public _Node<K, V> next;
        public K key;
        public V value;
    }

    private HashMap<K, _Node<K, V>> _queue;
    private _Node<K, V> _first;
    private _Node<K, V> _last;
}
