/* misc.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;

namespace Sugar {
    /** Similar to python range type */
    public struct Range {
        public int start;
        public int stop;
        public int step;

        public Range () {
            start = -1;
            stop = -1;
            step = 1;
        }

        public int last {
            get { return stop - step; }
            set { stop = value + step; }
        }

        public int size {
            get { return stop - start; }
        }

        public bool is_empty {
            get { return stop == start; }
        }

        public bool contains (int @value) {
            return @value >= start && @value < stop;
        }

        public bool is_equal (Range other) {
            return start == other.start && stop == other.stop;
        }

        public bool is_subset (Range other) {
            return other.contains (start) && other.contains (last);
        }
    }

    public struct Timestamp {
        public TimeVal tv;

        public Timestamp () {
            tv.get_current_time ();
        }

        public double diff (Timestamp other) {
            var d_sec = tv.tv_sec - other.tv.tv_sec;
            var d_usec = tv.tv_usec - other.tv.tv_usec;

            if ((d_sec >= 0) != (d_usec >= 0)) {
                var sign = (d_sec > 0 ? +1 : -1);
                d_sec -= sign;
                d_usec += sign * 1000000;
            }

            return (double) d_sec + (double) d_usec / 1000000.0;
        }
    }

    /** XXX no way to type cast char* to unowned string directly */
    public unowned string to_string (char *str) {
        return (string) str;
    }

    /** XXX no way to type cast char* to unowned uchar[] directly */
    public unowned uchar[] to_uchars (char *str) {
        return (uchar[]) str;
    }

    public class Trigger : Object {
        public signal void activate (Gtk.Widget initiator);

        public static unowned Trigger? find_dock (
                Type parent_type, Gtk.Widget parent, string prop_name) {
            if (parent.get_type ().is_a (parent_type)) {
                var trigger = Value (typeof (Object));
                parent.get_property (prop_name, ref trigger);
                return trigger.get_object () as Trigger;
            } else if (parent.parent != null) {
                return find_dock (parent_type, parent.parent, prop_name);
            } else {
                return null;
            }
        }
    }

    public Gdk.Color color_type_to_rgb (int type) {
        return color_to_rgb (Color.get (type));
    }

    public Gdk.Color color_to_rgb (Color color) {
        Gdk.Color gdk_color = { };
        gdk_color.red = color.red;
        gdk_color.green = color.green;
        gdk_color.blue = color.blue;
        return gdk_color;
    }

    /**
     * Get preview image of current activity Gdk window
     *
     * @return  array of bytes with PNG image or null if Environ.window
     *          was not set; useful to set Jobject.preview property
     */
    public Array<uchar > get_preview () throws Error {
        assert (Environ.window != 0);

        var window = Gdk.Window.foreign_new ((Gdk.NativeWindow) Environ.window);
        return_val_if_fail (window != null, null);

        int width, height;
        window.get_size (out width, out height);

        var pixbuf = Gdk.pixbuf_get_from_drawable (null, window, null,
                0, 0, 0, 0, width, height);
        return_val_if_fail (pixbuf != null, null);

        var preview = pixbuf.scale_simple (
                Environ.PREVIEW_WIDTH, Environ.PREVIEW_HEIGHT,
                Gdk.InterpType.BILINEAR);
        return_val_if_fail (pixbuf != null, null);

        Array<uchar> result = new Array<uchar> (false, false, 1);

        preview.save_to_callback ((buf, count) => {
                result.append_vals (buf, (uint) count);
                return true;
            }, "png");

        return (owned) result;
    }

    public string default_icon_path () {
        if (_default_icon_path != null)
            return _default_icon_path;

        // Force returning .svg
        var svg_size = 1024;
        var icon = Gtk.IconTheme.get_default ().lookup_icon (
                "application-octet-stream", svg_size,
                Gtk.IconLookupFlags.FORCE_SVG);
        assert (icon != null);
        _default_icon_path = icon.get_filename ();

        return _default_icon_path;
    }

    private string _default_icon_path;
}
