/* jobject.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * High level interface to journal entry
 *
 * To create new journal entry, call Journal.create to make empty local object,
 * then call Jobject.write to create journal entry on sugar-datastore side.
 * If journal entry already exists, use Journal.find.
 *
 * On every property change (or set_field invocation), Jobject won't send new
 * data to sugar-datastore, call Jobject.write. Properties could be changed
 * implicitly (if entry was changed in Journal activity for example), so,
 * connect to notification signal to reflect on such changes e.g. notify::title.
 */
public class Sugar.Jobject : Object {
    /**
     * Signal will be emited on receiving async results of Jobject.write call
     *
     * @param error if error was occured, error will contain error message,
     *              null otherwise
     */
    public signal void write_end (string? error);

    /**
     * Unique journal entry id
     *
     * Value could be null if jobject was never written to sugar-datastore.
     * This field can not be changed manually.
     */
    public string? uid {
        get {
            unowned string field_value = _get_string ("uid");
            return field_value[0] != '\0' ? field_value : null;
        }
        private set {
            _set_string ("uid", value);
        }
    }

    /**
     * Last time when jobject was saved by activity
     *
     * While changing metadata in Journal activity,
     * timestamp is not being changed.
     */
    public long timestamp {
        get {
            return long.parse (_get_string ("timestamp"));
        }
        set {
            _set_string ("timestamp", value.to_string ());
            // support Journal-0.82
            var time = Time.local (value);
            _set_string ("mtime", time.format ("%Y-%m-%dT%H:%M:%S"));
        }
    }

    /**
     * Title of journal entry
     */
    public string title {
        get {
            return _get_string ("title");
        }
        set {
            _set_string ("title", value);
        }
    }

    /**
     * Was title value set by user
     *
     * This value should be set to true after setting title in reaction on user
     * input e.g. after setting title from GUI entry widget.
     */
    public bool title_set_by_user {
        get {
            return int.parse (_get_string ("title_set_by_user")) != 0;
        }
        set {
            _set_string ("title_set_by_user", value ? "1" : "0");
        }
    }

    /**
     * MIME type of file associated with journal entry
     */
    public string mime_type {
        get {
            return _get_string ("mime_type");
        }
        set {
            _set_string ("mime_type", value);
        }
    }

    /**
     * Type of sharing of journal entry
     *
     * How object was shared.
     */
    public ShareScope share_scope {
        get {
            unowned string field_value = _get_string ("share-scope");
            for (int i = 0; i < ShareScope.LAST; i++)
                if (field_value == _share_scope_values[i])
                    return (ShareScope) i;
            return ShareScope.PRIVATE;
        }
        set {
            if (value < 0 || value >= ShareScope.LAST)
                value = ShareScope.PRIVATE;
            _set_string ("share-scope", _share_scope_values[value]);
        }
    }

    /**
     * The bundle_id of activity which created this object
     *
     * This value can't be empty or set explicitly, it will be initiated only in
     * Jobject.create.
     */
    public string activity {
        get {
            unowned string field_value = _get_string ("activity");
            if (field_value[0] != '\0')
                return field_value;
            else {
                warning ("Empty activity for %s jobject", uid);
                return "unknown.activity";
            }
        }
        set {
            _set_string ("activity", value);
        }
    }

    /**
     * The activity instance id which created this object
     *
     * This value can't be empty or set explicitly, it will be initiated only in
     * Jobject.create.
     */
    public string activity_id {
        get {
            unowned string field_value = _get_string ("activity_id");
            if (field_value[0] != '\0')
                return field_value;
            else {
                warning ("Empty activity_id for %s jobject", uid);
                return "0000000000000000000000000000000000000000";
            }
        }
        set {
            _set_string ("activity_id", value);
        }
    }

    /**
     * Was object stared by user
     */
    public bool keep {
        get {
            return int.parse (_get_string ("keep")) != 0;
        }
        set {
            _set_string ("keep", value ? "1" : "0");
        }
    }

    /**
     * Preview of lastly saved object
     *
     * Preview should be in PNG format.
     */
    public Array<uchar >?preview {
        get {
            unowned Value? preview_value = get_field ("preview");
            if (preview_value.holds (Type.BOXED))
                return (Array<uchar >?) preview_value.get_boxed ();
            else
                return null;
        }
        set {
            var preview_value = Value (Type.from_name ("GArray<uchar>"));
            preview_value.set_boxed (value);
            _set_field ("preview", preview_value);
        }
    }

    /**
     * Color pair of user who created this object
     *
     * If object was shared, this color could be not color of current user.
     */
    public XoColor icon_color {
        owned get {
            unowned string field_value = _get_string ("icon-color");
            if (field_value[0] != '\0')
                return XoColor.from_string (field_value);
            else {
                warning ("Jobject %s lacks of icon-color", uid);
                return Profile.color;
            }
        }
        set {
            _set_string ("icon-color", value.to_string ());
        }
    }

    /**
     * File that is associated with current Jobject
     *
     * @return  full path or null if there is no associated file
     *
     * Property returns path to the file that is stored by sugar-datastore.
     * The only valid operation is read. To change associated file,
     * use Jobject.write_file.
     */
    public string? file_path {
        get {
            if (_file_path == null && uid != null) {
                try {
                    _file_path = _get_ds ().get_filename (uid);
                } catch (Error error) {
                    warning ("Cannot get filename for %s: %s",
                            uid, error.message);
                }
            }
            return _file_path;
        }
    }

    private Jobject () {
        var metadata = new HashTable<string, Value?>.full (
                str_hash, str_equal, g_free, Value.unset);
        _metadata = metadata;
        _changed = new HashTable<string, bool> (str_hash, str_equal);
        _write_queue = new Queue<_WriteQueueItem?> ();
    }

    /**
     * Create new local object
     *
     * @return  new Jobject in all cases
     *
     * Function doesn't ask sugar-datastore, just creates new local
     * representation of journal entry. To send data to sugar-datastore,
     * use Jobject.write. To create representation of already exist journal
     * entry, use Jobject.find.
     */
    public static Jobject create () {
        var jobject = new Jobject ();

        jobject.title = _("%s Activity").printf (Environ.activity.name);
        jobject.title_set_by_user = false;
        jobject.activity = Environ.bundle_id;
        jobject.activity_id = Environ.activity_id;
        jobject.keep = false;
        jobject.share_scope = ShareScope.PRIVATE;
        jobject.icon_color = Profile.color;

        return jobject;
    }

    /**
     * Create representation of journal entry
     *
     * @param uid   unique id of journal entry
     * @return      Jobject or null on errors
     *
     * If function cannot connect to sugar-datastore or journal entry with
     * given uid does not exist, Error will be thrown.
     */
    public static Jobject? find (string uid) {
        try {
            var jobject = new Jobject ();
            jobject._get_properties (uid, false);
            return jobject;
        } catch (Error error) {
            warning ("Cannot find journal entry %s: %s", uid, error.message);
            return null;
        }
    }

    public override void dispose () {
        if (_updated_handler != 0) {
            try {
                Datastore.instance ().disconnect (_updated_handler);
            } catch (Error error) {
                // pass
            }
        }
        _unlink (_file_path);
        base.dispose ();
    }

    /**
     * Get field by name
     *
     * @param field_name    field name
     * @return              stored field value or null if field doesn't exist
     */
    public unowned Value? get_field (string field_name) {
        return _metadata.lookup (field_name);
    }

    /**
     * Set field by name
     *
     * @param field_name    field name
     * @param field_value   value to store
     *
     * Function will also emit notify signal for appropriate field property.
     */
    public void set_field (string field_name, Value field_value)
            requires (field_name != "uid") {
        _set_field (field_name, field_value);
        var prop_name = field_name.delimit ("_", '-');
        if (get_class ().find_property (prop_name) != null)
            notify_property (prop_name);
    }

    /**
     * Write metadata changes to sugar-datastore
     *
     * Method is async, write_end signal will be emited on receiving reply from
     * sugar-datastore. If Jobject was constructed by Jobject.create, new
     * journal entry will be created on sugar-datastore side and uid property
     * will be set after emiting write_end.
     *
     * There is no need in waiting write_end before second write invocation,
     * write will queue calls.
     *
     * If activity doesn't support GLib main loop, set Environ.sync_dbus to true to
     * make all time synchronous DBus calls.
     */
    public void write () {
        _write (null, false, false);
    }

    /**
     * Write metadata changes and file to sugar-datastore
     *
     * @param file_path             path to the file that should be associated
     *                              with journal entry, if null, current
     *                              associated file will be preserved
     * @param transfer_ownership    should sugar-datastore take ownership
     *                              of passed file_path
     *
     * Does the same that Jobject.write does. Also stores file that will be
     * associated with journal entry.
     *
     * If file_path argument was specified and transfer_ownership is true,
     * caller should not rely on existence of file_path after write call. On
     * successful invocation, file_path will be moved to new location, after
     * error, file will be deleted.
     *
     * There is no need in waiting write_end before second write invocation,
     * write will queue calls.
     */
    public void write_file (string? file_path, bool transfer_ownership = false) {
        _write (file_path, transfer_ownership, false);
    }

    /**
     * Initiate copy-on-write copy of current journal entry
     *
     * Does the same Jobject.write does and reset uid property to null,
     * thus second journal entry will appear only after followed write call.
     */
    public void cow () {
        _write (null, false, true);
    }

    /**
     * Delete journal entry
     *
     * @return  true if dbus call was successfully sent, otherwise false
     *
     * If newly create Jobject was never written, function does nothing.
     * Function makes async dbus call.
     *
     * If activity doesn't support GLib main loop, set Environ.sync_dbus to true to
     * make all time synchronous DBus calls.
     */
    public bool unlink () {
        if (uid == null)
            debug ("Jobject was never written, just discard it");
        else {
            debug ("Delete Jobject %s", uid);

            try {
                _get_ds ().@delete (uid, _delete_cb);
            } catch (Error error) {
                warning ("Cannot delete journal entry %s: %s", uid,
                        error.message);
                return false;
            }

            _unlink (_file_path);
            _file_path = null;
        }
        return true;
    }

    public unowned string _get_string (string field_name) {
        unowned Value? field_value = get_field (field_name);
        if (field_value != null && field_value.holds (typeof (string)))
            return field_value.get_string ();
        else
            return "";
    }

    public void _set_string (string field_name, string field_value) {
        var string_value = Value (typeof (string));
        string_value.set_string (field_value);
        _set_field (field_name, string_value);
    }

    public void _set_field (string field_name, Value field_value) {
        _metadata.replace (field_name, field_value);
        _changed.replace (field_name, true);
    }

    private dynamic DBus.Object _get_ds () throws Error {
        var ds = Datastore.instance ();

        if (_updated_handler == 0)
            _updated_handler = ds.updated.connect (_ds_updated_cb);

        return ds.service;
    }

    private void _write (string? file_path, bool transfer_ownership, bool cow) {
        var queue_item = _WriteQueueItem (file_path, transfer_ownership, cow);
        _write_queue.push_tail (queue_item);

        if (_write_queue.length > 1)
            debug ("Queue Jobject.write uid=%s file_path=%s transfer=%d",
                    uid, queue_item.file_path,
                    (int) queue_item.transfer_ownership);
        else
            _write_begin ();
    }

    private void _write_begin () {
        var queue_item = _write_queue.peek_head ();

        debug ("Jobject.write uid=%s file_path=%s transfer=%d",
                uid, queue_item.file_path, (int) queue_item.transfer_ownership);

        timestamp = time_t ();
        _changed.remove_all ();

        try {
            if (Environ.sync_dbus) {
                if (uid == null)
                    uid = _get_ds ().create (_metadata, queue_item.file_path,
                            queue_item.transfer_ownership);
                else
                    _get_ds ().update (uid, _metadata, queue_item.file_path,
                            queue_item.transfer_ownership);
                _write_end (null);
            } else {
                if (uid == null)
                    _get_ds ().create (_metadata, queue_item.file_path,
                            queue_item.transfer_ownership, _create_cb);
                else
                    _get_ds ().update (uid, _metadata, queue_item.file_path,
                            queue_item.transfer_ownership, _update_cb);
            }
        } catch (Error error) {
            _write_end (error);
        }
    }

    private void _write_end (Error? error) {
        var queue_item = _write_queue.pop_head ();

        if (error != null) {
            if (queue_item.unlink_on_success)
                _unlink (queue_item.file_path);
            warning ("Cannot write jobject %s: %s", uid, error.message);
        } else {
            if (queue_item.unlink_on_fail)
                _unlink (queue_item.file_path);
            if (queue_item.cow && _metadata.lookup ("uid") != null)
                _metadata.remove ("uid");
            if (queue_item.file_path[0] != '\0') {
                _unlink (_file_path);
                _file_path = null;
            }
        }

        if (_write_queue.is_empty ())
            write_end (error == null ? null : error.message);
        else
            _write_begin ();
    }

    private void _get_properties (string uid, bool on_update) throws Error {
        HashTable<string, Value?> metadata = _get_ds ().get_properties (uid);

        metadata.for_each ((key, src_value) => {
            unowned string field_name = (string) key;

            if (!on_update || !_changed.lookup (field_name)) {
                Value dst_value = { };
                var prop_name = field_name.delimit ("_", '-');

                if (src_value.holds (Type.BOXED)) {
                    if (get_class ().find_property (prop_name) != null &&
                            field_name != "preview") {
                        unowned Array<uchar> array_value =
                                (Array<uchar >?) src_value.get_boxed ();

                        string *str_value =
                                GLib.malloc (array_value.length + 1);
                        char *tr_value_p = (char *) str_value;
                        for (var i = array_value.length + 1; i-- > 0;)
                            tr_value_p[i] = (char) array_value.index (i);
                        tr_value_p[array_value.length] = '\0';

                        dst_value = Value (typeof (string));
                        dst_value.take_string ((owned) str_value);
                    } else {
                        dst_value = src_value;
                    }
                } else {
                    dst_value = Value (typeof (string));
                    src_value.transform (ref dst_value);
                }

                _metadata.replace (field_name, dst_value);
                if (on_update && get_class ().find_property (prop_name) != null)
                    notify_property (prop_name);
            }
        });

        /* datastore-0.82 does not return uid property */
        this.uid = uid;
    }

    private void _unlink (string? file_path) {
        if (file_path != null &&
                FileUtils.test (file_path, FileTest.EXISTS) &&
                FileUtils.unlink (file_path) != 0)
            warning ("Cannot remove file %s: %s", file_path, strerror (errno));
    }

    private void _create_cb (string uid, Error error) {
        if (error == null)
            this.uid = uid;
        _write_end (error);
    }

    private void _update_cb (Error error) {
        _write_end (error);
    }

    private void _delete_cb (Error error) {
        if (error != null)
            warning ("Cannot delete jobject entry: %s", error.message);
    }

    private void _ds_updated_cb (string uid) {
        if (uid != this.uid)
            return;

        debug ("Update jobject %s", uid);

        try {
            _get_properties (uid, true);
        } catch (Error error) {
            warning ("Cannot update Jobject %s: %s", uid, error.message);
        }
    }

    private struct _WriteQueueItem {
        public string file_path;
        public bool transfer_ownership;
        public bool cow;
        public bool unlink_on_success;
        public bool unlink_on_fail;

        public _WriteQueueItem (string? file_path, bool transfer_ownership,
                bool cow) {
            if (file_path == null) {
                this.file_path = "";
                transfer_ownership = false;
            } else if (!Path.is_absolute (file_path)) {
                this.file_path = Path.build_filename (
                        Environment.get_current_dir (), file_path);
            } else {
                this.file_path = file_path;
            }

            this.cow = cow;

            if (transfer_ownership && Environ.secure_mode) {
                /* TODO check for 0007 permitions for parent directory */
                warning ("There is no way to just move files to the " +
                        "datastore under rainbow without creating 0777 " +
                        "directories, thus have to copy file");
                this.transfer_ownership = false;
                this.unlink_on_success = true;
                this.unlink_on_fail = true;
            } else {
                this.transfer_ownership = transfer_ownership;
                this.unlink_on_success = false;
                this.unlink_on_fail = transfer_ownership;
            }
        }
    }

    private const string[] _share_scope_values = {
        /* ShareScope.PRIVATE */
        "private",
        /* ShareScope.INVITE_ONLY */
        "invite",
        /* ShareScope.PUBLIC */
        "public"
    };

    private ulong _updated_handler;
    private HashTable<string, Value?> _metadata;
    private HashTable<string, bool> _changed;
    private string? _file_path;
    private static Queue<_WriteQueueItem?> _write_queue;
}
