/* paletteconnector.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Port from original sugar-toolkit project.
 * File:   src/sugar/graphics/palettewindow.py
 * Commit: c684950ecff34e910e6da5840737a469f95a0e79
 *
 * Copyright (C) 2007, Eduardo Silva <edsiper@gmail.com>
 * Copyright (C) 2008, One Laptop Per Child
 * Copyright (C) 2009, Tomeu Vizoso
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

public enum Sugar.PalettePosition {
    ANCHORED,
    AT_CURSOR
}

public abstract class Sugar.Connector : Object {
    public signal void invoker_enter ();
    public signal void invoker_leave ();
    public signal void popup ();
    public signal void popdown ();

    public Object subject { get; set; }

    public PalettePosition palette_position {
        get; set; default = PalettePosition.ANCHORED;
    }

    /** Helper function to find the gap position and size of widget a */
    public static bool get_box_gap (Gdk.Rectangle a, Gdk.Rectangle b,
            out Gtk.PositionType gap_side, out int gap_start,
            out int gap_size) {
        if (a.y + a.height == b.y)
            gap_side = Gtk.PositionType.BOTTOM;
        else if (a.x + a.width == b.x)
            gap_side = Gtk.PositionType.RIGHT;
        else if (a.x == b.x + b.width)
            gap_side = Gtk.PositionType.LEFT;
        else if (a.y == b.y + b.height)
            gap_side = Gtk.PositionType.TOP;
        else
            return false;

        if (gap_side == Gtk.PositionType.BOTTOM ||
                gap_side == Gtk.PositionType.TOP) {
            gap_start = int.min (a.width, int.max (0, b.x - a.x));
            gap_size = int.max (0,
                    int.min (a.width, (b.x + b.width) - a.x) - gap_start);
        } else if (gap_side == Gtk.PositionType.RIGHT ||
                gap_side == Gtk.PositionType.LEFT) {
            gap_start = int.min (a.height, int.max (0, b.y - a.y));
            gap_size = int.max (0,
                    int.min (a.height, (b.y + b.height) - a.y) - gap_start);
        }

        return true;
    }

    public void get_palette_pos (Gtk.Requisition palette_size,
            out int x, out int y) {
        var alignment = _get_alignment (palette_size);
        var rect = _get_position_for_alignment (alignment, palette_size);

        if (_in_screen (rect)) {
            x = rect.x;
            y = rect.y;
        } else {
            // In case our efforts to find an optimum place inside the screen
            // failed, just make sure the palette fits inside the screen
            // if at all possible.
            x = int.max (0, rect.x);
            y = int.max (0, rect.y);

            x = int.min (x, Gdk.Screen.width () - rect.width);
            y = int.min (y, Gdk.Screen.height () - rect.height);
        }
    }

    public virtual void on_popup () {
    }

    public virtual void on_popdown () {
    }

    public abstract void draw_rectangle (Gdk.EventExpose event,
            PaletteWindow palette);

    public abstract unowned Gtk.Window get_toplevel ();

    public abstract Gdk.Rectangle get_rect ();

    protected virtual unowned Alignment[] get_alignments () {
        if (palette_position == PalettePosition.AT_CURSOR)
            return ALIGNMENT_AT_CURSOR;
        else
            return ALIGNMENT_ANCHORED;
    }

    private Gdk.Rectangle _get_position_for_alignment (Alignment alignment,
            Gtk.Requisition palette_size) {
        Gdk.Rectangle rect = { };

        if (palette_position == PalettePosition.ANCHORED)
            rect = get_rect ();
        else {
            int cursor_x, cursor_y;
            var display = Gdk.Display.get_default ();
            display.get_pointer (null, out cursor_x, out cursor_y, null);

            var dist = Metrics.get (Metrics.PALETTE_CURSOR_DISTANCE);
            rect.x = cursor_x - dist;
            rect.y = cursor_y - dist;
            rect.width = dist * 2;
            rect.height = dist * 2;
        }

        rect.x = (int) (rect.x + rect.width * alignment.invoker_halign +
                palette_size.width * alignment.palette_halign);
        rect.y = (int) (rect.y + rect.height * alignment.invoker_valign +
                palette_size.height * alignment.palette_valign);
        rect.width = palette_size.width;
        rect.height = palette_size.height;

        return rect;
    }

    private Alignment _get_alignment (Gtk.Requisition palette_size) {
        int best_area = -1;
        Alignment *best_alignment = null;

        foreach (var alignment in get_alignments ()) {
            var rect = _get_position_for_alignment (alignment, palette_size);
            if (_in_screen (rect))
                return alignment;
            int area = _get_area_in_screen (rect);
            if (area > best_area) {
                best_alignment = &alignment;
                best_area = area;
            }
        }

        Alignment result = *best_alignment;
        var rect = get_rect ();

        if (result.position == Gtk.PositionType.LEFT ||
                result.position == Gtk.PositionType.RIGHT) {
            int top = rect.y;
            int bottom = Gdk.Screen.height () - rect.y - rect.width;
            if (top > bottom)
                // Set palette_valign to align to screen on the top
                result.palette_valign = -(float) top / palette_size.height;
            else
                // Set palette_valign to align to screen on the bottom
                result.palette_valign = -(float)
                        (palette_size.height - bottom - rect.height) /
                        palette_size.height;
            result.invoker_valign = 0;
        } else if (result.position == Gtk.PositionType.TOP ||
                result.position == Gtk.PositionType.BOTTOM) {
            int left = rect.x;
            int right = Gdk.Screen.width () - rect.x - rect.width;
            if (left > right)
                // Set palette_halign to align to screen on left
                result.palette_halign = -(float) left / palette_size.width;
            else
                // Set palette_halign to align to screen on right
                result.palette_halign = -(float)
                        (palette_size.width - right - rect.width) /
                        palette_size.width;
            result.invoker_halign = 0;
        }

        return result;
    }

    /** Return area of rectangle visible in the screen */
    private int _get_area_in_screen (Gdk.Rectangle rect) {
        var x1 = int.max (0, rect.x);
        var y1 = int.max (0, rect.y);
        var x2 = int.min (Gdk.Screen.width (), rect.x + rect.width);
        var y2 = int.min (Gdk.Screen.height (), rect.y + rect.height);

        return (x2 - x1) * (y2 - y1);
    }

    private bool _in_screen (Gdk.Rectangle rect) {
        return rect.x >= 0 && rect.y >= 0 &&
                rect.x + rect.width <= Gdk.Screen.width () &&
                rect.y + rect.height <= Gdk.Screen.height ();
    }

    protected struct Alignment {
        public float palette_halign;
        public float palette_valign;
        public float invoker_halign;
        public float invoker_valign;
        public int position;
    }

    private const Alignment[] ALIGNMENT_AT_CURSOR = {
        { 0.0f, 0.0f, 1.0f, 1.0f, -1 },
        { 0.0f, -1.0f, 1.0f, 0.0f, -1 },
        { -1.0f, -1.0f, 0.0f, 0.0f, -1 },
        { -1.0f, 0.0f, 0.0f, 1.0f, -1 }
    };

    private const Alignment[] ALIGNMENT_ANCHORED = {
        { 0.0f, 0.0f, 0.0f, 1.0f, Gtk.PositionType.BOTTOM },
        { -1.0f, 0.0f, 1.0f, 1.0f, Gtk.PositionType.BOTTOM },
        { 0.0f, 0.0f, 1.0f, 0.0f, Gtk.PositionType.RIGHT },
        { 0.0f, -1.0f, 1.0f, 1.0f, Gtk.PositionType.RIGHT },
        { 0.0f, -1.0f, 0.0f, 0.0f, Gtk.PositionType.TOP },
        { -1.0f, -1.0f, 1.0f, 0.0f, Gtk.PositionType.TOP },
        { -1.0f, 0.0f, 0.0f, 0.0f, Gtk.PositionType.LEFT },
        { -1.0f, -1.0f, 0.0f, 1.0f, Gtk.PositionType.LEFT }
    };
}

public class Sugar.WidgetConnector : Sugar.Connector {
    public Gtk.Widget widget {
        get {
            return subject as Gtk.Widget;
        }
        set {
            if (widget != null) {
                widget.enter_notify_event.disconnect (_enter_notify_event_cb);
                widget.leave_notify_event.disconnect (_leave_notify_event_cb);
                widget.button_release_event.disconnect (
                        _button_release_event_cb);
                widget.hierarchy_changed.disconnect (_hierarchy_changed_cb);
            }

            subject = value;

            if (widget != null) {
                widget.enter_notify_event.connect (_enter_notify_event_cb);
                widget.leave_notify_event.connect (_leave_notify_event_cb);
                widget.button_release_event.connect (_button_release_event_cb);
                widget.hierarchy_changed.connect (_hierarchy_changed_cb);
            }
        }
    }

    public WidgetConnector (Gtk.Widget widget) {
        Object (widget: widget);
    }

    public override void draw_rectangle (Gdk.EventExpose event,
            PaletteWindow palette) {
        Gtk.PositionType gap_side;
        int gap_start, gap_size;
        int x = 0, y = 0;

        if (widget.is_no_window ()) {
            x = widget.allocation.x;
            y = widget.allocation.y;
        }

        if (get_box_gap (get_rect (), palette.get_rect (), out gap_side,
                        out gap_start, out gap_size))
            Gtk.paint_box_gap (widget.style, event.window,
                    Gtk.StateType.PRELIGHT, Gtk.ShadowType.IN, event.area,
                    widget, "palette-invoker", x, y,
                    widget.allocation.width, widget.allocation.height,
                    gap_side, gap_start, gap_size);
        else
            Gtk.paint_box (widget.style, event.window,
                    Gtk.StateType.PRELIGHT, Gtk.ShadowType.IN, event.area,
                    widget, "palette-invoker", x, y,
                    widget.allocation.width, widget.allocation.height);
    }

    public override void on_popup () {
        widget.queue_draw ();
    }

    public override void on_popdown () {
        widget.queue_draw ();
    }

    public override unowned Gtk.Window get_toplevel () {
        return widget.get_toplevel () as Gtk.Window;
    }

    public override Gdk.Rectangle get_rect () {
        Gdk.Rectangle result = {
            0, 0,
            widget.allocation.width, widget.allocation.height
        };

        if (widget.window != null)
            widget.window.get_origin (out result.x, out result.y);
        else
            warning ("Trying to position palette with invoker " +
                    "that's not realized.");

        if (widget.is_no_window ()) {
            result.x += widget.allocation.x;
            result.y += widget.allocation.y;
        }

        return result;
    }

    private bool _enter_notify_event_cb (Gdk.EventCrossing event) {
        invoker_enter ();
        return false;
    }

    private bool _leave_notify_event_cb (Gdk.EventCrossing event) {
        invoker_leave ();
        return false;
    }

    private bool _button_release_event_cb (Gdk.EventButton event) {
        if (event.button == 3) {
            popup ();
            return true;
        } else {
            return false;
        }
    }

    private void _hierarchy_changed_cb (Gtk.Widget? previous_toplevel) {
        popdown ();
    }
}

public class Sugar.ToolConnector : Sugar.WidgetConnector {
    public Gtk.ToolItem tool_item {
        get {
            return widget.parent as Gtk.ToolItem;
        }
        set {
            if (value == null)
                widget = null;
            else
                widget = value.child;
        }
    }

    public ToolConnector (Gtk.ToolItem tool_item) {
        Object (tool_item: tool_item);
    }

    protected override unowned Connector.Alignment[] get_alignments () {
        var toolbar = tool_item.parent as Gtk.Toolbar;
        if (toolbar == null)
            return base.get_alignments ();
        else if (toolbar.orientation == Gtk.Orientation.HORIZONTAL)
            return ALIGNMENT_HORIZONTAL;
        else
            return ALIGNMENT_VERTICAL;
    }

    private const Connector.Alignment[] ALIGNMENT_HORIZONTAL = {
        { 0.0f, 0.0f, 0.0f, 1.0f, Gtk.PositionType.BOTTOM },
        { -1.0f, 0.0f, 1.0f, 1.0f, Gtk.PositionType.BOTTOM },
        { 0.0f, -1.0f, 0.0f, 0.0f, Gtk.PositionType.TOP },
        { -1.0f, -1.0f, 1.0f, 0.0f, Gtk.PositionType.TOP }
    };

    private const Connector.Alignment[] ALIGNMENT_VERTICAL = {
        { -1.0f, 0.0f, 0.0f, 0.0f, Gtk.PositionType.LEFT },
        { -1.0f, -1.0f, 0.0f, 1.0f, Gtk.PositionType.LEFT },
        { 0.0f, 0.0f, 1.0f, 0.0f, Gtk.PositionType.RIGHT },
        { 0.0f, -1.0f, 1.0f, 1.0f, Gtk.PositionType.RIGHT }
    };
}
