# Copyright (C) 2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import logging

import dbus
import gobject


class DataSource(gobject.GObject):
    """ Base data source class.

    The class represents high level functionality for data sources. The exact
    data needs to be provided from backend subclasses:

        * DataSource.do_search, overload in subclass
        * DataSource.on_entries, call from subclass on getting search results
        * DataSource.on_changed, call on initial state and any data changes

    The view widgets workflow is:

        # Connect to `changed` signal to be informed about DataSource
          ready-to-use state and any data changes;
        # Connect to `entry` signal to get data entries;
        # Connect to `estimated` signal to handle sutations when estimated
          value of total number of data entries has been changed;
        # Call DataSource.request, e.g. from DataSource.changed handler,
          to start emitting DataSource.entry events.

    The class is designed to support only one view at the same time,
    see DataSource.request for details.

    """
    __gsignals__ = {
            'changed': (
                # Data was changed
                # After receiving this signal, views need to call request()
                # to fetch refreshed data entries.
                gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE,
                []),
            'entry': (
                # Data entry content
                # This kind of events will be emited after calling request()
                gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE,
                # (uid, offset, data)
                [str, int, gobject.TYPE_PYOBJECT]),
            'estimated': (
                # New `total` value has been estimated
                gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE,
                []),
            }

    def __init__(self, page_size=32, cache_pages=1, **kwargs):
        """ Constructor.

        :param page_size:
            Fetch data from backend will happen by one page, storing data
            in internal cache is also page multiplied.
        :param cache_pages:
            TODO Number of pages to cache

        """
        assert page_size
        assert cache_pages

        gobject.GObject.__init__(self, **kwargs)
        self._page_size = page_size
        self._query = None
        self._order = []
        self._group = None
        self._keys = {}
        self._total = 0
        self._cache = []
        self._cache_offset = 0
        self._continue_offset = 0
        self._continue_limit = 0

    def get_query(self):
        return self._query

    def set_query(self, value):
        if value != self._query:
            self._query = value
            self.on_changed()

    """ Query string in Xapian query format. """
    query = gobject.property(type=str,
            getter=get_query, setter=set_query)

    def get_order(self):
        return self._order

    def set_order(self, value):
        if value != self._order:
            self._order = value
            self.on_changed()

    """ List of keys to sort requested entries with +/- optional prefixes. """
    order = gobject.property(type=object,
            getter=get_order, setter=set_order)

    def get_group(self):
        return self._group

    def set_group(self, value):
        if value != self._group:
            self._group = value
            self.on_changed()

    """ Group requested entries by data key. """
    group = gobject.property(type=str,
            getter=get_group, setter=set_group)

    def get_keys(self):
        return self._keys

    def set_keys(self, value):
        if value != self._keys:
            self._keys = value
            self.on_changed()

    """ Query by exact values for data keys. """
    keys = gobject.property(type=object,
            getter=get_keys, setter=set_keys)

    def get_total(self):
        return self._total

    """ Estimated total number of data entries in the datasource. """
    total = gobject.property(type=int, getter=get_total)

    def request(self, offset, limit):
        """ Request for data entries.

        :param offset:
            the first entry offset
        :param limit:
            number of entries to request

        The result of calling this method is emitting `entry` events
        with requested data entries.

        The result data might be returned immediatelly from local cache either
        requested from data backend. In the last case, any new `request`
        call will suppress previous one. So, DataSource is not designed
        to support several views at the same time - with one view, it is fine
        to forget about previous call, e.g., while scrolling data in a view.

        """
        self._continue_offset = offset
        self._continue_limit = limit
        self._continue_request()

    def store(self, uid, data):
        pass

    def do_search(self, offset, limit):
        """ Process the request.

        :param offset:
            return data starting from offset
        :param limit:
            number of entries to return

        The function needs to be implemented in backend subclass.

        """
        assert not 'implemented'

    def on_entries(self, offset, total, entries):
        """ Process request results.

        :param offset:
            offset from the request
        :param total:
            total number of entries in the data source
        :param entries:
            array of result entries

        The function needs to be called from backend subclass on reply
        of DataSource.search request.

        """
        if total != self._total:
            self._total = total
            self.emit('estimated')

        for i, entry in enumerate(entries):
            self.emit('entry', entry.get('url'), offset + i, entry)

        self._continue_request()
        # TODO add received entiries to extisted cache
        self._cache = entries
        self._cache_offset = offset

    def on_changed(self):
        """ Data has changed.

        The function needs to be called from backend subclass on initial state
        and on any data changes.

        """
        del self._cache[:]
        self.emit('changed')

    def _continue_request(self):
        if not self._continue_limit:
            return

        if self._cache_offset <= self._continue_offset:
            for i in self._cache[self._continue_offset - self._cache_offset:]:
                if not self._continue_limit:
                    break
                self.emit('entry', i.get('url'), self._continue_offset, i)
                self._continue_offset += 1
                self._continue_limit -= 1
        else:
            last = self._continue_offset + self._continue_limit - 1
            if last <= self._cache_offset + len(self._cache) - 1:
                while last >= self._cache_offset and self._continue_limit > 0:
                    entry = self._cache[last - self._cache_offset]
                    self.emit('entry', entry.get('url'), last, entry)
                    last -= 1
                    self._continue_limit -= 1

        if self._continue_limit:
            # Request happens in pages only
            shift = self._continue_offset % self._page_size
            self._continue_offset -= shift
            self._continue_limit += shift
            shift = self._continue_limit % self._page_size
            if shift > 0:
                self._continue_limit += self._page_size - shift

            offset = self._continue_offset
            self._continue_offset += self._page_size
            self._continue_limit -= self._page_size
            self.do_search(offset, self._page_size)


class DBusDataSource(DataSource):

    def __init__(self, name, path, interface, **kwargs):
        DataSource.__init__(self, **kwargs)

        self._name = name
        self._path = path
        self._interface = interface
        self._source = None
        self._source_hid = None
        self._mountpoint = '/'

        self._set_source()

        dbus.SessionBus().add_signal_receiver(self.__NameOwnerChanged_cb,
                signal_name='NameOwnerChanged',
                dbus_interface='org.freedesktop.DBus')

    @property
    def mountpoint(self):
        return self._mountpoint

    @mountpoint.setter
    def mountpoint(self, value):
        if value != self._mountpoint:
            self._mountpoint = value
            self.on_changed()

    def store(self, uid, data):
        if self._source is None:
            return
        self._source.Store(self.mountpoint, uid, data)

    def do_search(self, offset, limit):
        if self._source is None:
            return

        request = self.keys.copy()
        if self.query:
            request['query'] = self.query
        if self.order:
            request['order'] = self.order
        if self.group:
            request['group'] = self.group
        request['offset'] = offset
        request['limit'] = limit

        def reply_cb(entries, total):
            self.on_entries(offset, total, entries)

        def error_cb(error):
            logging.warning('Cannot process request to %s: %s',
                    self._path, error)
            self.on_entries(offset, 0, [])

        self._source.Search(self._mountpoint, request,
                reply_handler=reply_cb, error_handler=error_cb,
                byte_arrays=True)

    def _set_source(self):
        obj = dbus.SessionBus().get_object(self._name, self._path)
        self._source = dbus.Interface(obj, self._interface)
        self._source_hid = self._source.connect_to_signal(
                'Changed', self.__Changed_cb)
        self.on_changed ();

    def __NameOwnerChanged_cb(self, name, old, new):
        if name != self._name:
            return
        if old and not new:
            logging.info('Data source %s is offline', self._path)
            self._source_hid.remove()
            self._source_hid = None
            self._source = None
            self.on_changed ();
        elif not old and new and self._source is None:
            logging.info('Data source %s is online', self._path)
            self._set_source()

    def __Changed_cb(self, mountpoint):
        if mountpoint == self.mountpoint:
            self.on_changed()
