# Copyright (C) 2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import gtk

from polyol import env, gui
from polyol.libjournal import datawidgets


class DataView(gui.HomogeneTable):

    def __init__(self, **kwargs):
        gui.HomogeneTable.__init__(self, **kwargs)
        self._datasource = None
        self.modify_bg(gtk.STATE_NORMAL,
                gui.color_type_to_rgb(env.COLOR_WHITE))
        self.connect('frame_scrolled', self.__frame_scrolled_cb)

    @property
    def datasource(self):
        return self._datasource

    @datasource.setter
    def datasource(self, value):
        if value is self._datasource:
            return

        if self._datasource is not None:
            self._datasource.disconnect_by_func(self.__entry_cb)
            self._datasource.disconnect_by_func(self.__changed_cb)
            self._datasource.disconnect_by_func(self.__estimated_cb)

        self._datasource = value
        self._datasource.connect('entry', self.__entry_cb)
        self._datasource.connect('changed', self.__changed_cb)
        self._datasource.connect('estimated', self.__estimated_cb)

        self.__changed_cb(None)

    def do_new_cell(self):
        cell_widget = self.do_new_data_cell()
        assert cell_widget is not None
        cell = _DataCell()
        cell.add(cell_widget)
        cell.datasource = self._datasource
        return cell

    def do_fill_cell(self, cell, index):
        # Data fill happen via frame_scrolled handler
        pass

    def do_highlight_cell(self, cell, selected):
        pass

    def _refill(self):
        if self._datasource is None:
            return
        self._datasource.request(
                self.props.frame_range.start, self.props.frame_size)

    def __estimated_cb(self, sender):
        self.props.cell_count = self._datasource.total

    def __changed_cb(self, sender):
        self._refill()

    def __frame_scrolled_cb(self, sender):
        self._refill()

    def __entry_cb(self, sender, uid, offset, entry):
        cell = self.get_cell(offset)
        if cell is None:
            return
        cell.datasource = self._datasource
        cell.fill_in(uid, entry)


class _DataCell(gui.PaintBin):

    def __init__(self):
        gui.PaintBin.__init__(self)
        self._data_widgets = []
        self._entry = None
        self.datasource = None
        self.props.radius = 0
        self.modify_bg(gtk.STATE_NORMAL,
                gui.color_type_to_rgb(env.COLOR_WHITE))
        self.connect('destroy', self.__destroy_cb)

    def add(self, child):
        gui.PaintBin.add(self, child)
        self._scan_data_widgets(child)

    def fill_in(self, uid, entry):
        self._uid = uid
        for i in self._data_widgets:
            i.data = entry
            i.update_data_field()

    def _scan_data_widgets(self, widget):
        if isinstance(widget, gtk.Container):
            widget.foreach(self._scan_data_widgets)
        if isinstance(widget, datawidgets.DataWidget):
            self._data_widgets.append(widget)
            widget.connect('data-changed', self.__data_changed_cb)

    def __data_changed_cb(self, sender, key, value):
        self.datasource.store(self._uid, {key: value})

    def __destroy_cb(self, sender):
        for i in self._data_widgets:
            i.disconnect_by_func(self.__data_changed_cb)
        del self._data_widgets[:]
